<?php

namespace App\Http\Controllers\Assets\Settings;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\ApiCredential;
use App\Services\Integrations\Tickets\Spoke\SpokeIntegration;

/**
 * SpokeApiCredentialsController
 */
class SpokeApiCredentialsController extends Controller
{

    /**
     * @var ApiCredential|null The API credential for the 'spoke' service.
     */
    protected $apiCredential;

    /**
     * Constructor for the class.
     *
     * @param SpokeIntegration $ticketIntegration An instance of SpokeIntegration used for ticket integration.
     */
    public function __construct(protected SpokeIntegration $ticketIntegration)
    {
    }

    /**
     * Sets the API credential for the 'spoke' service.
     *
     * @return void
     */
    private function setApiCredential()
    {
        $this->apiCredential = ApiCredential::where('slug', 'spoke')->first();
    }

    /**
     * index
     *
     * @return void
     */
    public function index()
    {
        $this->setApiCredential();

        $apiCredential = $this->apiCredential;

        if ($apiCredential == null) {
            return redirect('/spoke/create');
        }

        try {
            $this->ticketIntegration->getAuthorizedUser();
            $status = 1;
        } catch (\Exception $e) {
            $status = 0;
        }

        return view('settings.spoke.index', compact('apiCredential', 'status'));
    }

    /**
     * Edit
     *
     * @return void
     */
    public function edit()
    {
        $this->setApiCredential();

        $credential = $this->apiCredential;

        return view('settings.spoke.edit', compact('credential'));
    }

    /**
     * create
     *
     * @return void
     */
    public function create()
    {
        $this->setApiCredential();

        if ($this->apiCredential) {
            return redirect('spoke/' . $this->apiCredential->id . '/edit');
        }

        return view('settings.spoke.edit');
    }

    /**
     * update
     *
     * @param  mixed $request
     * @param  mixed $id
     * @return void
     */
    public function update(Request $request, $id)
    {
        if ($id) {
            $request->validate([
                'url' => ['required', 'url'],
            ]);
        } else {
            $request->validate([
                'key' => ['required'],
                'url' => ['required', 'url'],
            ]);
        }

        try {
            ApiCredential::updateorcreate(
                ['id' => $id],
                [
                    'api_name' => 'Spoke',
                    'url' => $request->url,
                    'key' => $request->key,
                    'slug' => 'spoke',
                ]
            );

            return redirect('/spoke');
        } catch (\Exception $e) {
            logMessage("Spoke credential update failed with error " . $e->getMessage(), 'error');

            return redirect('/spoke')->with('error', 'Operation failed!');
        }
    }
}
