<?php
namespace App\Http\Controllers\Assets\Settings;

use App\Http\Controllers\Controller;
use App\Http\Requests\Slack\SlackCredentialsRequest;
use App\Services\SoftwareLicense\ImportedAssets\SlackService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;

class SlackCredentialsController extends Controller
{

    private $service;

    /**
     * Constructor for SlackCredentialsController
     *
     * @param SlackService $service
     */
    public function __construct(SlackService $service)
    {
        $this->service = $service;
    }

    /**
     * If the user has already created the Software asset API credential, show the user the settings page.
     * Otherwise, redirect the user to the create page
     *
     * @return object View is being returned.
     */
    public function index()
    {
        $credentialsData =  $this->service->getCredentials();

        if (!$credentialsData) {
            return redirect()->route('slack.create');
        }

        $status = $this->service->checkConnection($credentialsData);

        return view('settings.slack.index', ['credentials' => $credentialsData,'status' => $status]);
    }

    /**
     * create the new software asset API credential
     *
     * @return object View is being returned
     */
    public function create()
    {
        if ($this->service->getCredentials()) {
            return redirect()->route('slack.index');
        }

        return view('settings.slack.create');
    }

    /**
     * Store the Slack Credentials in the API Credential table
     *
     * @param SlackCredentialsRequest $request
     *
     * @return object Redirecting to the proper page
     */
    public function store(SlackCredentialsRequest $request)
    {
        if ($this->service->getCredentials()) {
            return redirect()->route('slack.index')->with('error','Slack credentials already exist. ');
        }

        $response = $this->service->addCredentials($request);

        return $response ? redirect()->route('slack.index')->with('message', __('message.created'))
            : redirect()->route('slack.create')->with('error', 'Opertaion Failed');
    }

    /**
     * It takes the id of the credential, finds it in the database, and returns a view with the
     * credential
     *
     * @return object View is being returned.
     */
    public function edit()
    {
        $credentialsData = $this->service->getCredentials();

        if (!$credentialsData) {
            return redirect()->route('slack.create');
        }

        return view('settings.slack.edit', ['credential' => $credentialsData]);
    }

    /**
     * It updates the credentials of the software asset account.
     *
     * @param object Request $request
     *
     * @return object Redirecting to the proper page
     */
    public function update(Request $request)
    {
        $response = $this->service->updateCredentials($request);

        return $response ? redirect(route('slack.index'))->with('message', 'Credentials updated successfully') :
            redirect(route('slack.index'))->with('error', 'Something went wrong. Try again later');
    }

    /**
     * Establishing the connection with the added software asset account
     *
     * @return object Redirect to the proper page after connection
     */
    public function connect()
    {
        $connectWithExistingData = $this->service->setConnection();

        if ($connectWithExistingData) {
            return redirect(route('slack.index'))->with('message', 'Connected successfully.');
        }

        $authenticationURL = $this->service->getAuthenticationURL();

        return redirect($authenticationURL);
    }

    /**
     * A callback function which is called when the user is redirected from software asset after
     * authentication.
     *
     * @return object Redirect to the proper page after connection
     */
    public function callback()
    {
        $code = request('code') ?? '';

        $credentialsData = $this->service->getCredentials();

        if (!$credentialsData) {
            return redirect(route('slack.index'))->with('error', 'Credentials not available.');
        }

        $authorizationResponse = $this->service->generateAccessToken($code, $credentialsData);

        if (!$authorizationResponse) {

            return redirect(route('slack.index'))->with('error', 'Authorization failed');
        }

        $updateAccessToken = $this->service->updateAccessToken($authorizationResponse);

        return $updateAccessToken ? redirect(route('slack.index'))->with('message', 'Authorization successfully completed and token updated successfully')
             : redirect(route('slack.index'))->with('error', 'Something went wrong. Try again later');

    }

    /**
     * This function changes the integration status of software asset and enables/disables the software asset accordingly
     *
     * @return bool
     */
    public function integrationStatusChange()
    {
        return $this->service->integrationStatusChange();
    }

    /**
     * A function to sync the data from software asset to our database.
     *
     * @return object Redirect to the proper page after connection
     */
    public function manualSync()
    {
        Artisan::call('Slack:sync');
        $softwareAssetSyncStatus = Artisan::output();
        $response = json_decode($softwareAssetSyncStatus, true);

        if ($response['status'] == 'success') {
            return redirect(route('slack.index'))->with( 'message', $response['message']);
        }

        return redirect(route('slack.index'))->with('error', $response['message']);
    }

}