<?php

namespace App\Http\Controllers\Assets\Settings;

use App\Events\BulkUpdates;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\SpecialUpload;
use App\Models\SiteCode;
use App\Models\Region;
use App\Models\Country;
use App\Models\AssetHistory;
use Facades\App\Services\SiteCodeBulkUpload;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use Facades\App\Services\SiteCodeData;

class SiteCodeController extends Controller
{
    public function index()
    {
        $siteCodes = SiteCode::with('region', 'country');
        $limit = 50;
        $num = num_row(request('page'), $limit);

        if (request('searchText')) {
            $siteCodes = SiteCodeData::filter($siteCodes, request('searchText'));
        }
        $siteCodes = $siteCodes->orderBy('code', 'asc')->paginate($limit);

        $regions = Region::orderBy('name', 'asc')->pluck('name', 'id');
        $countries = Country::orderBy('name')->pluck('name', 'name');
        return view('settings.site-code.site-code', compact('siteCodes', 'regions', 'countries', 'num'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param Request $request The request object containing the data to store.
     * @throws Some_Exception_Class description of exception
     * @return RedirectResponse A redirect response to the index page with a success message.
     */
    public function store(Request $request)
    {

        $this->validate(request(), [
            'site_code'     => 'required|unique:site_codes,code',
            'region_id'     => 'required',
            'country_id'     => 'required',
        ]);

        $siteCode = SiteCode::create([
            'code'            => $request->site_code,
            'region_id'        => $request->region_id,
            'country_id'    => $request->country_id,
        ]);

        $description = __('history.SiteCodeCreated', [
            'site_code' => $siteCode->code
        ]);
        $assetHistory = [
            'user_id'         => Auth::id(),
            'action'         => 'siteCode_created',
            'description'     => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));

        return redirect(route('site-code.index'))->with('message', __('message.created'));
    }

    /**
     * Update the site code.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function update()
    {
        $siteCode = SiteCode::findOrFail(request('id'));
        $oldSiteCode = $siteCode->code;
        $oldRegion = $siteCode->region->name;
        $oldCountry = $siteCode->country->name;

        $countryId = (Country::where('name', request('country'))->first())->id;

        $siteCode->update([
            'code'             => request('site_code'),
            'region_id'     => request('region'),
            'country_id'     => $countryId,
        ]);

        $siteCode = SiteCode::findOrFail(request('id'));

        $newSiteCode = $siteCode->code;
        $newRegion = $siteCode->region->name;
        $newCountry = request('country');

        $description = '';
        if ($oldSiteCode != $newSiteCode) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Site Code',
                'oldValue'  => $oldSiteCode,
                'newValue'  => $newSiteCode,
            ]);
        }

        if ($oldRegion != $newRegion) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Region',
                'oldValue'  => $oldRegion,
                'newValue'  => $newRegion,
            ]);
        }

        if ($oldCountry != $newCountry) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Country',
                'oldValue'  => $oldCountry,
                'newValue'  => $newCountry,
            ]);
        }

        $description = __('history.SiteCodeUpdated', [
            'description' => $description
        ]);

        $assetHistory = [
            'user_id'         => Auth::id(),
            'action'         => 'siteCode_updated',
            'description'     => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));

        return response()->json('success');
    }

    /**
     * Deletes a site code.
     *
     * @throws ModelNotFoundException if the site code with the given ID is not found
     * @return JsonResponse a JSON response with the success message
     */
    public function destroy()
    {
        $siteCode = SiteCode::findOrFail(request('id'));
        $siteCode->delete();

        $description = __('history.SiteCodeDeleted', [
            'site_code' => $siteCode->code
        ]);

        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'siteCode_deleted',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));

        return response()->json(__('message.deleted'));
    }

    /**
     * Executes an AJAX request and returns a JSON response.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response.
     */
    public function ajax()
    {

        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }

    /**
     * Uploads a file to the server, stores it in the public/asset_bulk_upload
     * directory, and imports the data from the file into the site_code table
     * using the SiteCodeBulkUpload service.
     *
     * If the import is successful, it redirects back to the site-code.index route
     * with a success message. Otherwise, it redirects back with an error message.
     *
     * @param  \App\Http\Requests\SpecialUpload  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function upload(SpecialUpload $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $file   = $request->file('bulk_upload_asset');

        $extension  = strtolower($file->getClientOriginalExtension());
        $fileName   = 'site-code-items-' . date("m-d-y") . '-' . time() . '.' . $extension;
        $path       = $file->storeAs('public/asset_bulk_upload', $fileName);

        $path = storage_path('app/' . $path);
        $count = SiteCodeBulkUpload::importData($path);
        if ($count > 0) {
            return redirect(route('site-code.index'))->with('message', __('message.created'));
        }
        return redirect(route('site-code.index'))->with('error', 'Operation failed. Please check the template.');
    }

    /**
     * Retrieves the site code details.
     *
     * @return \Illuminate\Http\JsonResponse Returns a JSON response containing the region ID and country ID.
     */
    public function getSiteCodeDetails()
    {
        $siteCodeId = request('siteCode');
        $siteCode = SiteCode::find($siteCodeId);
        $regionId = $siteCode->region_id;
        $countryId = $siteCode->country_id;
        return response()->json(['region_id' => $regionId, 'country_id' => $countryId]);
    }
}
