<?php

namespace App\Http\Controllers\Assets\Settings;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\ApiCredential;
use Exception;
use Illuminate\Support\Facades\Cache;

class RevivnController extends Controller
{
    /**
     * Retrieves the first ApiCredential with a slug of 'revivn' and displays the 'settings.revivn.index' view if found.
     * Otherwise, redirects to the 'revivn.create' route.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        $credentials = ApiCredential::where('slug', 'revivn')->first();

        if ($credentials) {
            return view('settings.revivn.index', compact('credentials'));
        }

        return redirect(route('revivn.create'));
    }

    /**
     * Revivn credentials creation page
     */
    public function create()
    {
        return view('settings.revivn.create');
    }

    /**
     * Store the Revivn credentials.
     *
     * @param Request $request The HTTP request object.
     * @return \Illuminate\Http\RedirectResponse The redirect response.
     */
    public function store(Request $request)
    {
        $request->validate([
            'url' => ['required', 'url'],
            'user'   => 'required',
            'password'   => 'required',
        ]);
        Cache::forget('revivn-credentials');
        Cache::forget('revivn-access-token');

        ApiCredential::Create([
            'api_name' => 'Revivn',
            'slug' => 'revivn',
            'url' => add_slash_at_end($request->url),
            'user_name' => $request->user,
            'password' => $request->password,
        ]);

        return redirect('/revivn')->with('message', 'Revivn credentials added successfully.');
    }

    /**
     * Edit the Revivn credentials.
     *
     * @param int $id The ID of the credential to edit.
     * @throws Exception If the credential is not found.
     * @return \Illuminate\Contracts\View\View|\Illuminate\Http\RedirectResponse
     */
    public function edit($id)
    {
        try {
            $credential = ApiCredential::findOrFail($id);
            return view('settings.revivn.edit', compact('credential'));
        } catch (Exception) {
            return redirect(route('revivn.index'))->with('error', 'Something went wrong. Try again later');
        }
    }

    /**
     * Update the Revivn credentials.
     *
     * @param Request $request The HTTP request object.
     * @param int $id The ID of the credential to update.
     * @return \Illuminate\Http\RedirectResponse The redirect response.
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'url' => ['required', 'url'],
        ]);
        Cache::forget('revivn-credentials');
        Cache::forget('revivn-access-token');

        try {
            $credential = ApiCredential::findOrFail($id);
            $updateData = [
                'url'   => add_slash_at_end($request->url, "/"),
                'user_name' => $request->user ?: $credential->user_name,
                'password'  => $request->password ?: $credential->password,
            ];
            $credential->update($updateData);

            return redirect(route('revivn.index'))->with('message', 'Credentials updated successfully.');
        } catch (Exception $e) {
            return redirect(route('revivn.index'))->with('error', 'Something went wrong. Try again later.');
        }
    }
}
