<?php

namespace App\Http\Controllers\Assets\Settings;

use App\Events\BulkUpdates;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Region;
use App\Models\AssetHistory;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class RegionController extends Controller
{
    public function index()
    {
        $regions = Region::orderBy('name')->get();
        return view('settings.region.region', compact('regions'));
    }

    public function store()
    {
        $this->validate(request(), [
            'region' => 'required',
        ]);

        $region = request('region');
        $regionData = Region::where('name', $region)->first();
        if (!$regionData) {
            Region::create([
                'name' => $region,
            ]);

            $description = __('history.RegionCreated', [
                'region' => $region
            ]);
            $assetHistory = [
                'user_id' => Auth::id(),
                'action' => 'region_created',
                'description' => $description,
                'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            ];
            event(new BulkUpdates($assetHistory));
            return redirect('/region');
        } else {
            return redirect('/region')->withErrors(['Region already exists']);
        }
    }

    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }

    public function update()
    {
        $regionRow = Region::findOrFail(request('region_id'));
        $oldRegionValue = $regionRow->name;
        $regionRow->update(['name' => request('name')]);

        $description = __('history.RegionUpdated', [
            'oldRegionValue' => $oldRegionValue,
            'newRegionValue' => request('name')
        ]);
        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'region_updated',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));
        return response()->json('success');
    }

    public function destroy()
    {
        $region = Region::findOrFail(request('region_id'));
        $regionValue = $region->name;
        if (is_countable($region->locations) && count($region->locations)) {
            return response()->json('error');
        }
        $region->delete();
        $description = __('history.RegionDeleted', [
            'regionValue' => $regionValue
        ]);
        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'region_deleted',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));
        return response()->json("Deleted Successfully");
    }
}
