<?php

namespace App\Http\Controllers\Assets\Settings;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Asset;
use App\Models\AssetHistory;
use App\Models\PurgeAssetHistory;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class PurgeAssetController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $limit = 50;
        $num = num_row(request('page'), $limit);

        $assets = $this->filterAssets()->paginate($limit);
        return view('settings.purge-assets', compact('assets', 'num'));
    }

    /**
     * Destroys an asset.
     *
     * @param int $id The ID of the asset to be destroyed.
     * @throws Exception If something goes wrong during the asset deletion process.
     * @return \Illuminate\Http\RedirectResponse A redirect response with a success or error message.
     */
    public function destroy($id)
    {
        try {
            $asset = Asset::with('location')->findOrFail($id);
            if (optional($asset->location)->is_freeze == 1) {
                return redirect()->back()->with('error', 'The asset #' . $asset->serial_no . ' exists in a frozen location and can\'t be deleted');
            }
            $assetData = $this->getAssetData($asset);
            PurgeAssetHistory::create($assetData);
            $asset->delete();
            AssetHistory::where('asset_id', $id)->delete();
            return redirect()->back()->with('message', 'Asset deleted successfully');
        } catch (Exception $e) {
            Log::channel('daily')->error('Asset deletion error:' . $e->getMessage());

            return redirect()->back()->with('error', 'Something went wrong, Try again later');
        }
    }


    public function getAssetData($asset)
    {
        $assetData = [
            'user_id'           => Auth::id(),
            'asset_type_id'     => $asset->asset_type_id,
            'make_and_model_id' => $asset->make_and_model_id,
            'technical_spec_id' => $asset->technical_spec_id,
            'asset_tag'         => $asset->asset_tag,
            'serial_no'         => $asset->serial_no,
        ];
        return $assetData;
    }

    public function filterAssets()
    {
        $assets = Asset::select('id', 'asset_tag', 'serial_no', 'asset_type_id')->orderBy('asset_tag');

        if (request('asset_tag')) {
            $assets->where('asset_tag', 'LIKE', '%' . request('asset_tag') . '%');
        }

        if (request('serial_number')) {
            $assets->where('serial_no', 'LIKE', '%' . request('serial_number') . '%');
        }


        return $assets;
    }
}
