<?php

namespace App\Http\Controllers\Assets\Settings;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\ApiCredential;
use App\Http\Traits\GeneratesSecureToken;
use App\Models\UserType;
use Illuminate\Validation\ValidationException;

class OktaScimController extends Controller
{
    use GeneratesSecureToken;

    public function index()
    {
        $credential = ApiCredential::firstOrCreate([
            'slug' => 'okta_scim'
        ], [
            'api_name' => 'Okta SCIM',
            'integration_status' => 0,
            'url' => url('/api/scim/v2')
        ]);

        $userRoles = UserType::userRoles()->orderBy('name')->get();

        return view('settings.okta-scim.index', compact('credential', 'userRoles'));
    }

    public function update(Request $request, ApiCredential $oktaScim)
    {
        try {

            $this->validate($request, [
                'integration_status' => 'required|boolean',
                'user_type_id' => 'required|integer',
            ]);

            $data = [
                'integration_status' => (int)$request->integration_status,
            ];

            $credential = $oktaScim->fresh();

            if (!$credential->integration_status && $request->integration_status) {
                $data['key'] = $this->generateSecureToken(32);
            }

            $data['role_id'] = $request->user_type_id;

            $credential->update($data);

            return redirect()->route('okta-scim.index')->with('message', __('message.updated'));
        } catch (ValidationException $e) {
            return redirect()->route('okta-scim.index')->with('error', 'Invalid input data');
        } catch (\Exception $e) {
            return redirect()->route('okta-scim.index')->with('error', 'Something went wrong. Try again later');
        }
    }

    public function generateToken()
    {
        $secureToken = $this->generateSecureToken(32);

        ApiCredential::whereSlug('okta_scim')->update([
            'key' => $secureToken
        ]);

        return response()->json([
            'token' => $secureToken,
        ]);
    }
}
