<?php

namespace App\Http\Controllers\Assets\Settings;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\ApiCredential;
use Exception;
use Illuminate\Support\Facades\Artisan;

class OktaDirectoryController extends Controller
{
    public function index()
    {
        $credentials = ApiCredential::where('slug', 'okta_directory')->first();
        if ($credentials) {
            $status = 0;
            return view('settings.okta-directory.index', compact('credentials', 'status'));
        }

        return redirect()->route('okta-directory.create');
    }

    public function create()
    {
        $credential = ApiCredential::where('slug', 'okta_directory')->first();
        if ($credential) {
            return redirect('okta-directory/' . $credential->id . '/edit');
        }
        return view('settings.okta-directory.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'api_token' => ['required'],
            'issuer_url' => ['required', 'url'],
        ]);
        try {

            ApiCredential::create([
                'api_name'  => 'Okta Directory',
                'slug'      => 'okta_directory',
                'url'       => add_slash_at_end($request->issuer_url),
                'key'       => $request->api_token,
            ]);

            return redirect()->route('okta-directory.index')->with('message', __('message.created'));
        } catch (Exception $e) {
            return redirect(route('okta-directory.index'))->with('error', 'Something went wrong. Try again later');
        }
    }

    public function edit($id)
    {
        try {
            $credential = ApiCredential::findOrFail($id);
            return view('settings.okta-directory.edit', compact('credential'));
        } catch (Exception $e) {
            return redirect(route('okta-directory.index'))->with('error', 'Something went wrong. Try again later');
        }
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'issuer_url' => ['required', 'url'],
        ]);
        try {
            $credential = ApiCredential::findOrFail($id);
            $credential->update([
                'url'       => add_slash_at_end($request->issuer_url),
                'key'       => $request->api_token ?: $credential->key,
            ]);

            return redirect()->route('okta-directory.index')->with('message', __('message.updated'));
        } catch (Exception $e) {
            return redirect(route('okta-directory.index'))->with('error', 'Something went wrong. Try again later');
        }
    }

    /**
     * Triggers a manual synchronization of the Okta Directory data.
     *
     * This function uses the `OktaDirectory:oktadirectory` Artisan command to do the synchronization.
     * It sets unlimited execution time and memory limit before calling the command. The function catches any
     * exceptions that may occur during the synchronization process and logs the error. The function returns a JSON
     * response with the status (success, warning, or error) and a message describing the result of the synchronization
     * process.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the status and message of the
     *                                       synchronization process.
     */
    public function manualSync()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        try {
            Artisan::call('OktaDirectory:oktadirectory');
            $response = Artisan::output();
            return response()->json(array('status' => 'success', 'message' => $response));
        } catch (\Exception $e) {
            return response()->json(array('status' => 'error', 'message' => 'Okta Directory sync failed.'));
        }
    }
}
