<?php

namespace App\Http\Controllers\Assets\Settings;

use App\Events\BulkUpdates;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\LocationType;
use App\Models\AssetHistory;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class LocationTypeController extends Controller
{
    public function index()
    {
        $locationTypes = LocationType::orderBy('location_type_name')->get();
        return view('settings.location-type.location-type', compact('locationTypes'));
    }

    public function store()
    {
        try {
            $this->validate(request(), [
                'location_type' => 'required',
            ]);

            $locationType = request('location_type');
            $locationData = LocationType::where('location_type_name', $locationType)->first();
            if (!$locationData) {
                LocationType::create([
                    'location_type_name' => $locationType,
                ]);

                $description = __('history.LocationTypeCreated', [
                    'locationType' => $locationType
                ]);
                $assetHistory = [
                    'user_id' => Auth::id(),
                    'action' => 'locationType_created',
                    'description' => $description,
                    'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                    'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                ];
                event(new BulkUpdates($assetHistory));

                return redirect('/location-type');
            } else {
                return redirect('/location-type')->withErrors(['Location type already exists']);
            }
        } catch (Exception $e) {
            return redirect('/location-type')->withErrors(['Some Error Occured.  Please Try Again']);
        }
    }

    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }

    public function update()
    {
        try {
            $locationType = LocationType::findOrFail(request('location_type_id'));
            $oldLocationType = $locationType->location_type_name;
            $locationType->update([
                'location_type_name' => request('name'),
                'row' => request('row'),
                'rack' => request('rack'),
                'bin' => request('bin'),
            ]);

            $description = __('history.LocationTypeUpdated', [
                'oldLocationType' => $oldLocationType,
                'newLocationType' => request('name')
            ]);
            $assetHistory = [
                'user_id' => Auth::id(),
                'action' => 'locationType_updated',
                'description' => $description,
                'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            ];
            event(new BulkUpdates($assetHistory));

            return response()->json('success');
        } catch (Exception $e) {
            return response()->json('Some Error Occured.  Please Try Again');
        }
    }

    public function destroy()
    {
        try {
            $locationType = LocationType::findOrFail(request('location_type_id'));
            $locationTypeValue = $locationType->location_type_name;
            if (is_countable($locationType->locations) && count($locationType->locations)) {
                return response()->json('error');
            }
            $locationType->delete();

            $description = __('history.LocationTypeDeleted', [
                'locationType' => $locationTypeValue
            ]);
            $assetHistory = [
                'user_id' => Auth::id(),
                'action' => 'locationType_deleted',
                'description' => $description,
                'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            ];
            event(new BulkUpdates($assetHistory));

            return response()->json("Deleted Successfully");
        } catch (Exception $e) {
            return response()->json('Some Error Occured.  Please Try Again');
        }
    }
}
