<?php

namespace App\Http\Controllers\Assets\Settings;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\ApiCredential;
use Facades\App\Services\JiraService;
use Facades\App\Services\Integrations\Tickets\Jira\OAuthWrapper;
use Artisan;

class JiraCredentialsController extends Controller
{

    private $apiCredential;
    private $tokens;

    public function __construct()
    {
        $this->apiCredential = ApiCredential::where('slug', 'jira')->first();
        $this->tokens = [
            'oauth_token' => optional($this->apiCredential)->project,
            'oauth_token_secret' => optional($this->apiCredential)->key
        ];
    }

    public function index()
    {
        if (!$this->apiCredential) {
            return redirect('jira-credentials/create');
        }

        try {
            OAuthWrapper::getAuthorizedUser($this->tokens);
            $status = 1;
        } catch (\Exception $e) {
            $status = 0;
        }

        $apiCredential = $this->apiCredential;

        return view('settings.jira.index', compact('apiCredential', 'status'));
    }

    public function create()
    {
        if ($this->apiCredential) {
            return redirect('jira-credentials/' . $this->apiCredential->id . '/edit');
        }

        return view('settings.jira.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'url' => ['required', 'url'],
            'consumer_key' => ['required'],
        ]);

        try {
            ApiCredential::create([
                'api_name'  => 'Jira',
                'slug'      => 'jira',
                'url'       => $request->url,
                'password' => $request->consumer_key,
            ]);

            return redirect()->route('jira-credentials.index')->with('message', __('message.created'));
        } catch (\Exception $e) {
            return redirect(route('jira-credentials.index'))->with('error', 'Something went wrong. Try again later.');
        }
    }

    public function edit($id)
    {
        $credential = ApiCredential::find($id);

        try {
            OAuthWrapper::getAuthorizedUser($this->tokens);
            $status = 1;
        } catch (\Exception $e) {
            $status = 0;
        }

        return view('settings.jira.edit', compact('credential', 'status'));
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'url' => ['required', 'url'],
        ]);

        $credential = ApiCredential::findOrFail($id);
        $credential->update([
            'url' => $request->url,
            'password' => $request->consumer_key,
        ]);
        session()->forget('jiraConnectionValid');
        Artisan::call('queue:restart');

        return redirect('/jira-credentials');
    }

    public function connect()
    {
        try {
            $tempTokens = OAuthWrapper::login(route('jira-credentials.callback'));
            session(['tempTokens' => $tempTokens]);

            return redirect(OAuthWrapper::getAuthorizeUrl($tempTokens['oauth_token']));
        } catch (\Exception $e) {
            return redirect('/jira-credentials')->with('error', 'Please verify correct credentials');
        }
    }

    public function callback()
    {
        try {
            $tempTokens = session('tempTokens');
            $verifier = request('oauth_verifier');
            $token = OAuthWrapper::requestAccessToken($tempTokens, $verifier);
            $this->apiCredential->project = $token['oauth_token'];
            $this->apiCredential->key     = $token['oauth_token_secret'];
            $this->apiCredential->save();

            return redirect('/jira-credentials');
        } catch (\Exception $e) {
            return redirect('/jira-credentials')->with('error', 'Something went wrong!');
        }
    }


    /**
     * Download a public key file securely.
     *
     * This method ensures secure access by implementing a **whitelist approach** to 
     * allow only specific, pre-approved public key files. It prevents **directory traversal** 
     * attacks by not allowing arbitrary filenames.
     *
     * @param  string  $fileName  The requested filename (must match an entry in the whitelist).
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     *
     * @throws \Symfony\Component\HttpKernel\Exception\HttpException 403  If access is denied.
     * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException 404  If the file does not exist.
     */
    public function downloadPublicKey($fileName)
    {
        $fileName = preg_replace('/[^a-zA-Z0-9_-]/', '', str_replace(' ', '_', trim($fileName)));

        // Only allow specific, safe filenames (whitelist approach).
        $allowedFiles = [
            'jira_key' => 'jira_publickey.pem'
        ];

        // Validate requested file against the whitelist (Fix: Use array_key_exists).
        if (!array_key_exists($fileName, $allowedFiles)) {
            Log::warning("Unauthorized key download attempt: " . $fileName);
            abort(403, "Access Denied.");
        }

        $path = storage_path('keys/' . $allowedFiles[$fileName]);

        // Double-check the file exists.
        if (!file_exists($path)) {
            abort(404, "File Not Found.");
        }

        // Force the correct MIME type for security.
        return response()->download($path, null, [
            'Content-Type' => 'application/x-pem-file',
            'Content-Disposition' => 'attachment; filename="' . $fileName . '"'
        ]);
    }
}
