<?php

namespace App\Http\Controllers\Assets\Settings;

use Illuminate\Http\Request;
use App\Models\ApiCredential;
use Carbon\Exceptions\Exception;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Artisan;
use App\Services\Integrations\UserDirectory\HiBobIntegration;

class HiBobController extends Controller
{

    public function __construct(protected HiBobIntegration $service) {}

    /**
     * Show the credentials
     */
    public function index()
    {
        $credentials = $this->service->getApiCredentials();
        if ($credentials) {
            $status = 0;
            if (session('hibob-token')) {
                $status = 1;
            }
            return view('settings.hibob.index', compact('credentials', 'status'));
        }

        return redirect(route('hibob.create'));
    }

    /**
     * Create credential
     */
    public function create()
    {
        return view('settings.hibob.create');
    }

    /**
     * Store new API credentials.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'user_name' => 'required',
            'password' => 'required',
        ]);

        ApiCredential::create([
            'api_name' => 'HiBob',
            'slug' => 'hibob',
            'user_name' => $validatedData['user_name'],
            'password' => $validatedData['password'],
        ]);

        return redirect('/hibob')
            ->with('success', 'Credentials saved successfully');
    }

    /**
     * Edit the credential
     */
    public function edit($id)
    {
        $credential = ApiCredential::find($id);
        return view('settings.hibob.edit', compact('credential'));
    }

    /**
     * Update the specified API credential.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     * @throws \Illuminate\Database\Eloquent\ModelNotFoundException
     */
    public function update(Request $request, $id)
    {
        $validatedData = $request->validate([
            'user_name' => 'required',
            'password' => 'sometimes|nullable',
        ]);

        $credential = ApiCredential::findOrFail($id);

        $updateData = [
            'user_name' => $validatedData['user_name'],
        ];

        if (!empty($validatedData['password'])) {
            $updateData['password'] = $validatedData['password'];
        }

        $credential->update($updateData);

        return redirect('/hibob')->with('success', 'Credentials updated successfully');
    }

    /**
     * Connect the credential
     */
    public function connect()
    {
        try {
            $status = $this->service->ping();
            if ($status) {
                session()->put('hibob-token', "connected");
                return redirect('/hibob')->with('message', 'Connection successful');
            } else {
                session()->forget('ad-token');
                return redirect('/hibob')->with('error', 'Connection error');
            }
        } catch (\Exception $e) {
            return redirect('/hibob')->with('error', 'Connection error');
        }
    }

    /**
     * processing Manual Sync 
     */
    public function manualSync()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        try {
            $connection = $this->service->ping();
            if ($connection) {

                Artisan::call('HiBobUser:import');

                return response()->json(array('status' => 'success', 'message' => Artisan::output()));
            }

            return response()->json(array('status' => 'error', 'message' => 'Connection failed'));
        } catch (Exception $e) {
            return response()->json(array('status' => 'error', 'message' => 'Something went wrong. Try again later'));
        }
    }
}
