<?php

namespace App\Http\Controllers\Assets\Settings;

use App\Events\BulkUpdates;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\MakeAndModel;
use Facades\App\Models\Manufacturer;
use App\Models\AssetType;
use App\Services\Settings\HardwareStandardService as HardwareStandardService;
use Illuminate\Support\Facades\Auth;

class HardwareStandardController extends Controller
{
    private $hardwareStandardService;

    public function __construct(HardwareStandardService $hardwareStandardService)
    {
        $this->hardwareStandardService = $hardwareStandardService;
    }

    public function index()
    {
        $assettypes = AssetType::allWithoutComputerAccessories()->pluck('name', 'name');
        $manufacturers = Manufacturer::orderBy('slug')->pluck('name', 'name');
        $hardwareStandards = $this->hardwareStandardService->getHardwareStandards()->paginate(config('pagination.per_page'));

        $status = $hardwareStandards->map(
            function ($item) {
                return collect($item)->only(['id', 'status'])->all();
            }
        );

        $status = json_encode($status);

        return view('settings.hardware-standard.hardware-standard', compact('hardwareStandards', 'assettypes', 'status', 'manufacturers'));
    }

    /**
     * Carries out the search operation
     * @return [type]
     */
    public function search(Request $request)
    {
        if ($request->ajax()) {

            $pageLength = request('page_legth') ?? config('pagination.per_page');
            $sortColumns = [
                '2' => 'asset_types.name',
                '3' => 'manufacturers.name',
                '4' => 'name',
                '5' => 'assets_count',
            ];

            $sortColumn = $sortColumns[request('sort_column')] ?? 'name';
            $sortDirection = request('sort_direction') ?? 'asc';

            $hardwareStandards = $this->hardwareStandardService->getHardwareStandards();
            $hardwareStandards = $this->hardwareStandardService->filterHardwareStandards($hardwareStandards)->orderBy($sortColumn, $sortDirection)->paginate($pageLength);

            $view['hardware'] = view('settings.hardware-standard.hardware-standard-data', compact('hardwareStandards'))->render();

            $view['links'] = view('settings.hardware-standard.hardware-standard-pagination-links', compact('hardwareStandards'))->render();

            // Create the statuses collection of the hardware standards to show the tick mark on  the table.
            $status = $hardwareStandards->map(
                function ($item) {
                    return collect($item)->only(
                        [
                            'id',
                            'status'
                        ]
                    )->all();
                }
            );

            $view['status'] = $status;

            return response()->json($view);
        }
    }

    /**
     * Creates new hw standard
     * @return [type]
     */
    public function store()
    {
        $this->validate(request(), [
            'manufacturer' => 'required',
            'make_model' => 'required',
            'assettype' => 'required'
        ]);

        try {
            $manufacturer = request('manufacturer');
            $manufacturer_id = Manufacturer::updateOrCreate(
                ['slug' => str_slug($manufacturer, '_')],
                ['name' => $manufacturer]
            )->id;

            $assettype_id = request('assettype');
            $assetType = AssetType::where('id', $assettype_id)->first();

            $makeModel = request('make_model');

            $hardwareData = MakeAndModel::where('manufacturer_id', $manufacturer_id)
                ->where('asset_type_id', $assettype_id)
                ->where('name', $makeModel)
                ->first();

            if (!$hardwareData) {
                MakeAndModel::UpdateOrCreate([
                    'manufacturer_id' => $manufacturer_id,
                    'slug' => str_slug($makeModel, '_'),
                    'asset_type_id' => $assettype_id,
                    'name' => $makeModel
                ]);

                $description = __('history.HardwareStandardCreated', [
                    'assetType' => $assetType->name,
                    'manufacturer' => request('manufacturer'),
                    'make_model' => request('make_model')
                ]);
                $assetHistory = [
                    'user_id' => Auth::id(),
                    'action' => 'hardware_standard_created',
                    'description' => $description
                ];

                event(new BulkUpdates($assetHistory));

                return redirect('/hardware-standard')->with('message', 'Hardware standard created successfully.');
            } else {
                return redirect('/hardware-standard')->with('error', 'Hardware standard already exists.');
            }
        } catch (\Exception $e) {
            return redirect('/hardware-standard')->with('error', 'Some error occurred.');
        }
    }

    /**
     * Updates the data based on the given request.
     *
     * @param Request $request The request object containing the data to be updated.
     *
     * @return JsonResponse Returns a JSON response with either 'success' or 'failed' based on the success of the update operation.
     */
    public function update(Request $request)
    {
        if ($this->hardwareStandardService->updateHardwareStandard($request)) {
            return response()->json('success');
        }

        return response()->json('failed');
    }

    /**
     * Manages status change of make and models
     * @return [type]
     */
    public function updateStatus()
    {

        $makeAndModel = MakeAndModel::findOrFail(request('make_and_model_id'));

        $description = '';

        if ($makeAndModel->status == 0) {
            $makeAndModel->update(['status' => 1]);
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Status',
                'oldValue'    => 'Inactive',
                'newValue'  => 'Active',
            ]);
        } else {
            $makeAndModel->update(['status' => 0]);
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Status',
                'oldValue'    => 'Active',
                'newValue'  => 'Inactive',
            ]);
        }

        $description = __('history.HardwareStandardUpdated', [
            'description' => $description
        ]);

        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'hardware_standard_updated',
            'description' => $description
        ];

        event(new BulkUpdates($assetHistory));

        $makeAndModel = MakeAndModel::findOrFail(request('make_and_model_id'));
        $status = $makeAndModel->status;

        $array['success'] = 'success';
        $array['status']   = $status;
        $array['message'] = 'Status updated successfully.';

        return response()->json($array);
    }

    /**
     * Handles an AJAX request.
     *
     * @param Request $request The request object.
     * @return mixed The response of the AJAX request.
     */
    public function ajax(Request $request)
    {
        if (request('action') == 'edit') {
            return $this->update($request);
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        } elseif (request('action') == 'status') {
            return $this->updateStatus();
        }

        return response()->json(true);
    }

    /**
     * Hardware Standard Export
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $data = [];
        $datas = collect();
        $hardwares = $this->hardwareStandardService->getHardwareStandards();
        $hardwares = $this->hardwareStandardService->filterHardwareStandards($hardwares);
        $hardwares->chunk(5000, function ($hardwaresChunks) use ($data, $datas) {
            $datas->push($this->hardwareStandardService->getHardwareStandardData($hardwaresChunks, 0, $data));
        });

        return exportToCsv($datas->toArray());
    }

    /**
     * Deletes harware standard after checking wheater it has asets or not
     * @return [type]
     */
    public function destroy()
    {
        if (!$this->hardwareStandardService->deleteHardwareStandard(request('make_and_model_id'))) {
            return response()->json('error');
        }

        return response()->json("deleted");
    }

    /**
     * Bulk deletes harware standards after checking wheater it has asets or not
     * @return [type]
     */
    public function bulkDestroy()
    {
        $hardwareStandards   = MakeAndModel::whereIn('id', request('hardware_ids'))->get();

        foreach ($hardwareStandards as $hardware) {
            $this->hardwareStandardService->deleteHardwareStandard($hardware->id);
        }

        return response()->json("deleted");
    }
}
