<?php

namespace App\Http\Controllers\Assets\Settings;

use App\Events\BulkUpdates;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Asset;
use App\Models\AssetStatus;
use Carbon\Carbon;
use Storage;
use Illuminate\Support\Facades\Log;
use Exception;
use Illuminate\Support\Facades\Auth;
use App\Http\Requests\Eol\IndividualCertificateRequest;

class EolCertificatesController extends Controller
{
    protected $endOfLifeWipedStatusId;
    protected $updated;
    protected $notFound;
    protected $alreadyExist;

    public function __construct()
    {
        $this->endOfLifeWipedStatusId = AssetStatus::getStatusId('end_of_life_data_wiped')->first()->id;
        $this->updated = 0;
        $this->notFound = '';
        $this->alreadyExist = '';
    }

    /**
     * Show the certificate upload form
     *
     * @return Illuminate\Http\Response
     */
    public function index()
    {
        return view('settings.eol-certificates-upload');
    }

    /**
     * Handle the upload of individual certificate files.
     *
     * This method processes multiple certificate files, stores them in the designated folder, 
     * retrieves corresponding asset data, and updates the asset with the certificate details.
     *
     * @param \App\Http\Requests\Eol\IndividualCertificateRequest $request
     *        The validated request containing the uploaded files.
     *
     * @return mixed
     *         The response message containing the summary of the upload.
     */
    public function upload(IndividualCertificateRequest $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        try {
            $files      = $request->file('file');
            $folder     = $this->createFolder();

            foreach ($files as $file) {
                $filename = $file->getClientOriginalName();
                $file->storeAs($folder, $filename);

                $item = $this->getItemData(sanitizeFilename($filename));
                $asset = Asset::with('assetStatus')->getBySerialNumber($item["NormalizedSerialNumber"])->first();

                if ($asset) {
                    $certificateFileName = $item["CertificateFileName"];

                    if ($certificateFileName != false) {
                        $this->updateAssetData($asset, $item, $certificateFileName);
                        $this->updated++;
                    } else {
                        $this->alreadyExist .= '#' . $item["NormalizedSerialNumber"] . ', ';
                    }
                } else {
                    Log::channel('single')->info("EOL certificate Upload - Asset With Serial  # = " . $item["NormalizedSerialNumber"] . " does not exist");

                    $this->notFound .= '#' . $item["NormalizedSerialNumber"] . ', ';
                }
            }
        } catch (Exception $e) {
            Log::channel('single')->info("EOL certificate Upload Error - " . $e->getMessage());
        }

        return $this->createResponseMessage();
    }

    /**
     * Create new folder, if not exists
     */
    public function createFolder()
    {
        $folderName = date('Y-m-d');
        $path = '/public/certificates/' . $folderName;
        Storage::makeDirectory($path);

        return $path;
    }

    /**
     * Taking the asset details from the file name
     *
     * @param string $file
     * @return Array
     */
    public function getItemData($file)
    {
        $explodedFileName = explode('_', $file);
        $folderName     = date('Y-m-d');

        return [
            'NormalizedSerialNumber' => $explodedFileName[0],
            'CertificateDate' => Carbon::now()->format('Y-m-d'),
            'CertificateFileName' => 'certificates/' . $folderName . '/' . $file,
        ];
    }

    /**
     * Update asset details
     *
     * @param App\Models\Asset $asset
     * @param Array $item    Asset details
     */
    public function updateAssetData($asset, $item, $certificateFileName)
    {
        if ($asset) {
            $this->createAssetHistory($asset);

            $asset->update([
                'certificate_added_date' => $item['CertificateDate'],
                'ewaste_certificate' => $certificateFileName,
                'location_id' => null,
                'asset_status_id' => $this->endOfLifeWipedStatusId,
                'loaner_return_date' => null,
                'loaner_retention_date' => null,
                'user_id' => null,
                'lost_date' => null,
            ]);
        }
    }

    /**
     * Create asset history for eol certificate upload
     *
     * @param App\Models\Asset $asset
     */
    public function createAssetHistory($asset)
    {
        $assetHistory = [
            'asset_id' => $asset->id,
            'user_id' => Auth::id(),
            'action' => 'certificate_uploaded',
            'old_asset_status_id' => $asset->asset_status_id,
            'new_asset_status_id' => $this->endOfLifeWipedStatusId,
            'description' => __('history.CertificateUploaded', [
                'asset_name' => $asset->asset_tag,
                'asset_id' => $asset->id
            ]),
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        event(new BulkUpdates($assetHistory));
    }

    /**
     * Create json string with certificate file name for saving to DB
     *
     * @param App\Models\Asset $asset
     * @param String $certificate    Certificate file name
     */
    public function jsonFormattedCertificate($asset, $certificate)
    {
        if ($asset->ewaste_certificate) {

            isJSON($asset->ewaste_certificate) ? $existCertificates =  json_decode($asset->ewaste_certificate, true) : $existCertificates[0] = $asset->ewaste_certificate;
            $index = is_countable($existCertificates) ? count($existCertificates) : 0;

            if (!in_array($certificate, $existCertificates)) {
                $existCertificates[$index + 1] = $certificate;
            } else {
                return false;
            }
        } else {
            $existCertificates[0] = $certificate;
        }

        return json_encode($existCertificates);
    }

    /**
     *Making json response with message
     */
    public function createResponseMessage()
    {
        Log::channel('single')->info("EOL Certificate Upload - Total certificates added - " . $this->updated);

        $response['error_message'] = '';
        $response['success_message'] = '<div class="alert alert-success" role="alert">' . $this->updated . ' Certificates Updated</div>';

        if ($this->notFound) {
            $response['error_message']   = '<div class="alert alert-danger" role="alert">Assets with serial # ' . rtrim($this->notFound, ', ') . ' does not exist</div>';;
        }

        if ($this->alreadyExist) {
            $response['error_message']   .= '<div class="alert alert-warning" role="alert">Assets with serial # ' . rtrim($this->alreadyExist, ', ') . 'already has the given certificate</div>';;
        }

        return $response;
    }
}
