<?php

namespace App\Http\Controllers\Assets\Settings;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreDiscoveryToolsHardwareMapping;
use App\Http\Responses\ReportJsonResponse;
use App\Models\DiscoveryToolsHardwareMapping;
use App\Services\HardwareMapping\DiscoveryToolsHarwareMappingService;
use App\Services\HardwareMapping\HardwareMappingDataService;
use Facades\App\Repositories\HardwareStandards;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class DiscoveryToolsHardwareMappingController extends Controller
{
    public function __construct(protected DiscoveryToolsHarwareMappingService $mappingService, protected HardwareMappingDataService $hardwareMappingService)
    {
    }

    public function index()
    {
        $mappingData = $this->mappingService->getMappingData();
        $hardwareStandards = HardwareStandards::getHardwareStandardsWithoutAccessories();
        $mappingDevices = $this->mappingService->getMappingDevices();
        $mappedDevices = $this->mappingService->getMappedDevices();

        extract($mappingData);
        extract($mappingDevices);
        extract($mappedDevices);

        return view('settings.discovery-tools-hardware-mapping.index', compact(
            'teqtivityHardwares',
            'hardwares',
            'intuneHardwares',
            'chromeHardwares',
            'jamfHardwares',
            'workspaceOneHardwares',
            'kandjiHardwares',
            'mobileIronHardwares',
            'discoveryToolMappingData',
            'intuneMappedHardwares',
            'chromeMappedHardwares',
            'jamfMappedHardwares',
            'workspaceOneMappedHardwares',
            'kandjiMappedHardwares',
            'mobileIronMappedHardwares',
            'hardwareStandards',
            'chromeComputer',
            'intuneComputer',
            'jamfComputer',
            'workspaceOneComputer',
            'kandjiComputer',
            'mobileIrons',
            'mappedChrome',
            'mappedIntune',
            'mappedJamf',
            'mappedWorkspaceOne',
            'mappedKandji',
            'mappedMobileIrons'
        ));
    }

    /**
     * Taking the data to the Discovery hardwre mapping
     *
     * @param Illuminate\Http\Request $request
     * @return Http\Response
     */
    public function data(Request $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $filteredData = $this->hardwareMappingService->data();
        $teqtivityHardwares = $filteredData['teqtivityHardwares'];

        $start = request('start');
        $data = [];

        if (!empty($teqtivityHardwares)) {
            $data = $this->hardwareMappingService->getReportData($teqtivityHardwares, $start, $data);
        }

        return new ReportJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }

    public function search()
    {
        $filteredMappingData = $this->mappingService->getMappingSearchData();
        extract($filteredMappingData);
        $view['hardware'] = view('settings.discovery-tools-hardware-mapping.partials.hardware-mapping-data', compact('teqtivityHardwares'))->render();

        return response()->json($view);
    }

    /** Map Hardware devices
     * @return [type]
     */
    public function mapping()
    {
        $this->mappingService->mapIntune();
        $this->mappingService->mapJamf();
        $this->mappingService->mapChromebook();
        $this->mappingService->mapWorkspaceOne();
        $this->mappingService->mapKandji();
        $this->mappingService->mapMobileIron();

        return redirect('discovery-tools-hardware-mapping')->with('message', 'ReSynced Successfully');
    }

    public function store(StoreDiscoveryToolsHardwareMapping $request)
    {
        foreach ($request->except(['_token', 'make_and_model']) as $key => $value) {
            if (!$value) {
                continue;
            }

            DiscoveryToolsHardwareMapping::Create([
                'make_and_model_id' => request('make_and_model'),
                'mdm_name' => $key,
                'mdm_model' => $value,
            ]);
        }

        return redirect('/discovery-tools-hardware-mapping')->withMessage('Discovery tools hardware mapping added successfully.');
    }


    /**
     * Update the specified discovery tools hardware mapping.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success message.
     */
    public function update()
    {
        request()->validate([
            'id'    => ['required'],
            'make_and_model_id'    => ['required'],
            'mdm_name'    => ['required'],
            'mdm_model'    => ['required', Rule::unique('discovery_tools_hardware_mappings', 'mdm_model')
                ->where(function ($query) {
                    $query->where('make_and_model_id', request('make_and_model_id'))
                        ->where('mdm_name', request('mdm_name'));
                })
                ->ignore(request('id'))],
        ]);

        $discoveryToolsMapping = DiscoveryToolsHardwareMapping::findOrFail(request('id'));

        $discoveryToolsMapping->update([
            'make_and_model_id' => request('make_and_model_id'),
            'mdm_name' => request('mdm_name'),
            'mdm_model' => request('mdm_model'),
        ]);

        return redirect('/discovery-tools-hardware-mapping')->withMessage('Discovery tools hardware mapping updated successfully.');
    }

    public function destroy()
    {
        request()->validate([
            'id' => ['required']
        ]);

        $discoveryToolsHardwareMapping = DiscoveryToolsHardwareMapping::findOrFail(request('id'));
        $discoveryToolsHardwareMapping->delete();

        return redirect('/discovery-tools-hardware-mapping')->withMessage('Discovery tools hardware mapping deleted successfully.');
    }

    /**
     * Gets the hardware model from all MDM s based on search
     * @return [type]
     */
    public function getFilterHardwareMappingFields()
    {
        $search = request('search');

        if (!empty($search)) {

            $result = $this->hardwareMappingService->getFilterHardwareMappingFields($search);

            return response()->json($result);
        }
    }
}
