<?php

namespace App\Http\Controllers\Assets\Settings\DiscoveryTools;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\ApiCredential;
use App\Services\Integrations\MobileIronApi;
use Exception;
use Illuminate\Support\Facades\Artisan;

/**
 *  Controller Class for MobileIron API Credentials
 */
class MobileIronController extends Controller
{
    public function __construct(protected MobileIronApi $mobileIronApi)
    {
    }


    /**
     * Home page for mobile iron in settings
     * @return [type]
     */
    public function index()
    {

        $credentials = ApiCredential::where('slug', 'mobile_iron')->first();

        if (!$credentials) {
            return redirect(route('mobile-iron.create'));
        }
        try {
            $status = $this->mobileIronApi->checkApiConnection();
        } catch (\Exception $e) {
            $status = 0;
        }

        return view('settings.discovery-tools.mobile-iron.mobile-iron-credentials', compact('credentials', 'status'));
    }

    /**
     * loads create page
     * @return [type]
     */
    public function create()
    {
        $credential = ApiCredential::where('slug', 'mobile_iron')->first();
        if ($credential) {
            return redirect('mobile-iron/' . $credential->id . '/edit');
        }
        return view('settings.discovery-tools.mobile-iron.mobile-iron-credentials-create');
    }

    /**
     * Store the MobileIron API credentials 
     * @param Request $request
     * 
     * @return [type]
     */
    public function store(Request $request)
    {
        $request->validate([
            'url' => ['required', 'url'],
            'user_name' => ['required'],
            'password' => ['required'],
        ]);

        ApiCredential::create([
            'api_name' => 'MobileIron',
            'slug' => 'mobile_iron',
            'url' => $request->url,
            'user_name' => $request->user_name,
            'password' => $request->password,
        ]);

        return redirect('/mobile-iron');
    }

    /**
     * loads the edit page
     * @param mixed $id
     * 
     * @return [type]
     */
    public function edit($id)
    {
        try {
            $credential = ApiCredential::findOrFail($id);
            return view('settings.discovery-tools.mobile-iron.mobile-iron-credentials-edit', compact('credential'));
        } catch (Exception $e) {
            return redirect(route('mobile-iron.index'))->with('error', 'Something went wrong. Try again later');
        }
    }

    /**
     * Updates the API credentials 
     * @param Request $request
     * @param mixed $id
     * 
     * @return [type]
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'url' => ['required', 'url'],
            'user_name' => ['required'],
        ]);

        try {
            $credential = ApiCredential::findOrFail($id);
            $updateData = [
                'url'       => $request->url,
                'user_name' => $request->user_name,
                'password'  => $request->password ? $request->password : $credential->password,
            ];
            $credential->update($updateData);
            return redirect(route('mobile-iron.index'))->with('message', 'Credentials updated successfully');
        } catch (Exception $e) {
            return redirect(route('mobile-iron.index'))->with('error', 'Something went wrong. Try again later');
        }
    }

    /**
     * Import mobile iron devices using console command
     * @return [type]
     */
    public function manualSync()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        Artisan::call('MobileIronUpload:upload');

        return redirect()->route('mobile-iron.index')->with('message', 'MobileIron manual sync finished!');
    }
}
