<?php

namespace App\Http\Controllers\Assets\Settings\DiscoveryTools;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\ApiCredential;
use Crypt;
use App\Services\Integrations\IntuneIntegration;
use Illuminate\Support\Facades\Artisan;
use Microsoft\Graph\Graph;
use Microsoft\Graph\Model;

class IntuneController extends Controller
{
    protected $intuneService;

    public function __construct(IntuneIntegration $intuneService)
    {
        $this->intuneService = $intuneService;
    }

    /**
     * Show the credentials listing
     * 
     * @return View
     */
    public function index()
    {
        $credentials = ApiCredential::where('slug', 'intune')->first();
        if ($credentials) {
            $status = 0;

            return view('settings.discovery-tools.intune.index', compact('credentials', 'status'));
        }

        return redirect()->route('intune.create');
    }

    /**
     * Show the form for create new credentials
     * 
     * @return View
     */
    public function create()
    {
        $credential = ApiCredential::where('slug', 'intune')->first();
        if ($credential) {
            return redirect('intune/' . $credential->id . '/edit');
        }
        return view('settings.discovery-tools.intune.create');
    }

    /**
     * Save the new credentails
     * 
     * @param Request $request
     * 
     * @return View
     */
    public function store(Request $request)
    {
        $request->validate([
            'client_id' => 'required',
            'client_secret' => 'required',
        ]);

        ApiCredential::create([
            'api_name'  => 'Intune',
            'slug'      => 'intune',
            'url'       => $request->tenant,
            'user_name' => $request->client_id,
            'password'  => $request->client_secret,
        ]);

        return redirect()->route('intune.index')->with('message', __('message.created'));
    }

    /**
     * Show the form for edit the credential
     * 
     * @param int $id    Credentials iD
     * 
     * @return View
     */
    public function edit($id)
    {
        try {
            $credential = ApiCredential::findOrFail($id);
            return view('settings.discovery-tools.intune.edit', compact('credential'));
        } catch (Exception $e) {
            return redirect(route('intune.index'))->with('error', 'Something went wrong. Try again later');
        }
    }

    /**
     * Update the credentials
     * 
     * @param Request $request
     * @param int $id  Cedential ID
     * 
     * @return View
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'client_id' => ['required'],
        ]);
        try {
            $credential = ApiCredential::findOrFail($id);

            $credential->update([
                'url'       => $request->tenant,
                'user_name' => $request->client_id,
                'password'  => $request->client_secret ? $request->client_secret : $credential->password,
            ]);

            return redirect()->route('intune.index')->with('message', __('message.updated'));
        } catch (Exception $e) {
            return redirect(route('intune.index'))->with('error', 'Something went wrong. Try again later');
        }
    }

    /**
     * Connect the intune credentials
     */
    public function connect()
    {
        $authUrl = $this->intuneService->getAuthUrl();

        return redirect()->away($authUrl);
    }

    /**
     * Call back action From Intune Application
     */
    public function callback(Request $request)
    {
        // Validate state
        $expectedState = session('oauthState');
        $request->session()->forget('oauthState');
        $providedState = $request->query('state');

        if (!isset($expectedState)) {
            return redirect('/intune');
        }

        if (!isset($providedState) || $expectedState != $providedState) {
            return redirect('/intune')
                ->with('error', 'Invalid auth state')
                ->with('errorDetail', 'The provided auth state did not match the expected value');
        }

        // Authorization code should be in the "code" query param
        $authCode = $request->query('code');
        if (isset($authCode)) {
            try {
                // Make the token request
                $accessToken = $this->intuneService->getAccessToken($authCode);

                $graph = new Graph();
                $graph->setAccessToken($accessToken->getToken());

                $this->intuneService->storeTokens($accessToken);
            } catch (League\OAuth2\Client\Provider\Exception\IdentityProviderException $e) {
                return redirect('/intune')
                    ->with('error', 'Error requesting access token')
                    ->with('errorDetail', $e->getMessage());
            }
        }

        return redirect('/intune')
            ->with('error', $request->query('error'))
            ->with('errorDetail', $request->query('error_description'));
    }

    /**
     * Signout the connection
     */
    public function signout()
    {
        $this->intuneService->clearTokens();

        return redirect('/intune');
    }

    /**
     * Test the connection
     */
    public function test()
    {
        try {
            $response = $this->intuneService->getWindowsManagedDevices();

            if (isset($response['error'])) {
                return redirect('/intune')
                    ->with('error', "Error in Connection")
                    ->with('errorDetail', $response['error']['error']['message']);
            }

            if (isset($response['token_error'])) {
                return redirect('/intune')
                    ->with('error', "Error in Connection")
                    ->with('errorDetail', $response['token_error']);
            }

            return redirect('/intune')->with('message', 'Connection is successful.');
        } catch (Exception $e) {
            return redirect('/intune')->with('error', 'Something went wrong. Try again later');
        }
    }

    /**
     * manualSync
     *
     * @return void
     */
    public function manualSync()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        
        try {
            $credential = ApiCredential::where('slug', 'intune')->first();
            if (isset($credential) && $credential->data) {
                Artisan::call('IntuneDataSync:run');
                $message = Artisan::output();
                if (substr($message, 0, 22) == "Intune sync Completed.") {
                    return response()->json(array('status' => 'success', 'message' => $message));
                } else {
                    return response()->json(array('status' => 'error', 'message' => "Some Error Occured"));
                }
            }
            return response()->json(array('status' => 'error', 'message' => 'Connection failed'));
            // return redirect()->route('intune.index')->with('message', 'Intune manual sync finished!');
        } catch (Exception $e) {
            return response()->json(array('status' => 'error', 'message' => 'Something went wrong. Try again later'));
        }
    }
}
