<?php

namespace App\Http\Controllers\Assets\Settings\DiscoveryTools;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\GoogleWorkspaceClientSecretRequest;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use App\Models\ApiCredential;
use App\Services\Integrations\GoogleWorkspace\GoogleWorkspaceIntegration;
use Illuminate\Support\Facades\Artisan;

/**
 * Controller class for the Chrome Book api
 */
class ChromeBookController extends Controller
{
    public function __construct(protected GoogleWorkspaceIntegration $integration) {}

    /**
     * Show the credentials details
     */
    public function index()
    {
        $status = 0;
        try {
            $client = $this->integration->getAccessTokenWithJwt('chromebook');

            if ($client) {
                $status = 1;
            }
        } catch (\Exception $e) {
        }

        return view('settings.discovery-tools.chromebook.index', compact('status'));
    }

    /**
     * Show the credentials edit page
     */
    public function editCredentials()
    {
        $credential = ApiCredential::whereSlug('chromebook')->first();
        return view('settings.discovery-tools.chromebook.chromebook-credential-upload', compact('credential'));
    }

    /**
     * Save the Chromebook credentials. it might be a json file
     *
     * @param GoogleWorkspaceClientSecretRequest request The request object.
     */
    public function saveCredentials(GoogleWorkspaceClientSecretRequest $request)
    {
        if (getFileExtension(request()->file('client_secrete')->getClientOriginalName()) != 'json') {
            return redirect()->route('chrome-devices-edit')->withErrors('Client secret file must be json file');
        }
        $path = storage_path('keys');
        if (!File::isDirectory($path)) {
            File::makeDirectory($path, 0755, true, true);
        }
        try {
            Storage::disk('keys')->put('chromebook_client_secret.json', file_get_contents(request()->file('client_secrete')));
            $oldTockenFilePath = 'chromebook_client_secret_generated.json';

            /* If already generated accesstoken exist then rename the existing file.
            (This is for reset the existing connection.)  */

            if (file_exists(storage_path('keys/' . $oldTockenFilePath))) {
                $newTockenFilePath = Carbon::now()->format('m-d-Y_H-i-s') . '_chromebook_client_secret_generated.json';
                Storage::disk('keys')->move($oldTockenFilePath, $newTockenFilePath);
            }

            //Update the Email address in API Credentials table
            $data = [
                'api_name' => 'Chromebook',
                'slug' => 'chromebook',
            ];

            ApiCredential::updateOrCreate($data, ['user_name' => $request->admin_email]);
        } catch (Exception $e) {
            Log::channel('single')->info('Chromebook credential upload - ' . $e->getMessage());
            return redirect('/chrome-devices')->with('error', 'Something went wrong. Please try again later.');
        }
        return redirect('/chrome-devices')->with('message', 'Updated successfully.');
    }

    /**
     * Syncs the data manually.
     *
     * @throws \Exception Connection error. Please check the credentials
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message
     */
    public function manualSync()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        try {
            $client = $this->integration->getAccessTokenWithJwt('chromebook');
            if (!$client) {
                return response()->json(array('status' => 'error', 'message' => 'Connection failed.'));
            }
            Artisan::call('Chromebook:sync');
            return response()->json(array('status' => 'success', 'message' => Artisan::output()));
        } catch (\Exception $e) {
            return response()->json(array('status' => 'error', 'message' => 'Connection error. Please check the credentials.'));
        }
    }
}
