<?php

namespace App\Http\Controllers\Assets\Settings;

use App\Events\BulkUpdates;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Carrier;
use App\Models\Asset;
use App\Models\AssetHistory;
use App\Models\Country;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class CarrierController extends Controller
{
    public function index()
    {
        $carriers = Carrier::with('country')->orderBy('slug')->get();
        $countries = Country::orderBy('name')->pluck('name', 'name');
        return view('settings.mobile-carrier.mobile-carrier', compact('carriers', 'countries'));
    }

    public function store()
    {
        $this->validate(request(), [
            'carrier' => 'required',
        ]);

        $carrier = request('carrier');
        $carrierData = Carrier::where('slug', str_slug($carrier, '_'))->first();
        if (!$carrierData) {
            Carrier::create([
                'name' => $carrier,
                'slug' => str_slug($carrier, '_'),
                'country_id' => request('country_id') ?? null,
            ]);

            $description = __('history.MobileCarrierCreated', [
                'mobileCarrier' => $carrier
            ]);
            $assetHistory = [
                'user_id' => Auth::id(),
                'action' => 'mobileCarrier_created',
                'description' => $description,
                'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            ];
            event(new BulkUpdates($assetHistory));

            return redirect('/carrier');
        } else {
            return redirect('/carrier')->withErrors(['Mobile carrier already exists']);
        }
    }

    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }

    public function update()
    {
        $carrier = Carrier::findOrFail(request('carrier_id'));
        $oldCarrierValue = $carrier->name;
        $countryId = (Country::where('name', request('country'))->first())->id;
        $carrier->update(['name' => request('name'), 'slug' => str_slug(request('name'), '_'), 'country_id' => $countryId]);

        $description = __('history.MobileCarrierUpdated', [
            'oldCarrierValue' => $oldCarrierValue,
            'newCarrierValue' => request('name')
        ]);
        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'mobileCarrier_updated',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));
        return response()->json('success');
    }

    public function destroy()
    {
        $carrier = Carrier::findOrFail(request('carrier_id'));
        $carrierValue = $carrier->name;
        if (count($carrier->assets)) {
            return response()->json('error');
        }
        $carrier->delete();
        $description = __('history.MobileCarrierDeleted', [
            'carrierValue' => $carrierValue
        ]);
        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'mobileCarrier_deleted',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));

        return response()->json("Deleted Successfully");
    }
}
