<?php

namespace App\Http\Controllers\Assets\Settings;

use App\Http\Controllers\Controller;
use App\Http\Requests\SpecialUpload;
use App\Services\BulkUpload\BulkStatusUpdate;
use App\Services\BulkUpload\SpecialBulkStatusUpdate;
use Illuminate\Http\Request;
use App\Http\Requests\CsvFileUploadRequest;
use App\Http\Traits\FileSanitizationTrait;
use Illuminate\Support\Facades\Storage;

class BulkStatusUpdateController extends Controller
{
    use FileSanitizationTrait;

    /**
     * Initialize the controller
     *
     * @param BulkStatusUpdate $bulkStatusUpdate
     * @param SpecialBulkStatusUpdate $specialBulkStatusUpdate
     */
    public function __construct(protected BulkStatusUpdate $bulkStatusUpdate, protected SpecialBulkStatusUpdate $specialBulkStatusUpdate) {}

    public function index()
    {
        session(['asset_count' => 0]);

        return view('settings.bulk-status-update');
    }

    /**
     * Handle the upload of a CSV file for bulk asset updates.
     *
     * This method sets unlimited execution time and memory limit for large file processing,
     * retrieves the uploaded CSV file, saves it to a specific directory, and processes it
     * using the bulkStatusUpdate service.
     *
     * @param \App\Http\Requests\CsvFileUploadRequest $request
     *        The validated HTTP request containing the uploaded file.
     *
     * @return mixed
     *         The response from the bulkStatusUpdate service after processing the file.
     */
    public function upload(CsvFileUploadRequest $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        // Get the uploaded file from the request
        $file = $request->file('file');

        // Get the file extension in lowercase
        $extension = strtolower($file->getClientOriginalExtension());

        $fileName = 'update-asset-items-' . date('m-d-y') . '-' . time()  . '.' . $extension;

        $path = request()->file('file')->storeAs('public/asset_bulk_update', $fileName);

        return $this->bulkStatusUpdate->getViewData(storage_path('app/' . $path), $fileName);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $ticketNo = '';
        $filePath = '';
        $fileName = '';
        $errors = [];
        $updatedCount = 0;

        if ($request->count < 1) {
            return redirect(route('bulk-status-update.index'))->with('error', 'Assets cannot be updated.');
        }

        $basePath = storage_path('app/public/asset_bulk_update');

        for ($i = 1; $i <= $request->count; ++$i) {
            if (!$request->has('asset_id' . $i)) {
                continue; //row was deleted before submitting
            }

            $requestData = json_decode(request('asset_id' . $i), true);

            $ticketNo = $ticketNo ?: ($requestData['ticket_no'] ?? null);
            $filePath = $filePath ?: $this->sanitizeFilePath($requestData['filePath'], $basePath);
            $fileName = $fileName ?: $this->sanitizeFileName($requestData['fileName']);

            $updateResponse = $this->bulkStatusUpdate->saveData($request->all(), $i);
            if ($updateResponse === false) {
                $errors[] = json_decode(request('asset_id' . $i))->serial_no;
                continue;
            }

            $updatedCount++;
        }

        $this->bulkStatusUpdate->attachFiletoTicket($ticketNo, $filePath, $fileName);
        if (empty($errors)) {
            return redirect(route('bulk-status-update.index'))->with('message', 'Assets updated successfully.');
        }

        if ($request->count != count($errors)) {
            return redirect(route('bulk-status-update.index'))
                ->with([
                    'message' => "Total {$updatedCount} assets updated successfully.",
                    'error' => 'Assets ' . implode(', ', $errors) . ' cannot be updated.',
                ]);
        }

        return redirect(route('bulk-status-update.index'))->with('error', 'Assets cannot be updated.');
    }

    /**
     * Handles the special upload for bulk asset status updates.
     *
     * This function sets unlimited execution time and memory to handle large
     * file uploads. It saves the uploaded file in the public/asset_bulk_update
     * directory with a unique filename and processes the file using the
     * specialBulkStatusUpdate service to import asset data.
     *
     * If validation errors occur during processing, they are logged in a
     * timestamped file in the public/asset_bulk_update directory, and the
     * function redirects back to the bulk-status-update.index route with
     * the error details. If other errors occur, they are logged similarly,
     * and the function redirects back with a message indicating partial success
     * along with the error details.
     *
     * On successful completion without errors, the function redirects back
     * to the bulk-status-update.index route with a success message.
     *
     * @param \App\Http\Requests\SpecialUpload $request
     *        The validated request containing the uploaded file for bulk processing.
     *
     * @return \Illuminate\Http\RedirectResponse
     *         A redirect response indicating the outcome of the bulk upload process.
     */
    public function specialUpload(SpecialUpload $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $file   = $request->file('bulk_upload_asset');

        $extension  = strtolower($file->getClientOriginalExtension());
        $fileName   = 'update-asset-items-' . date("m-d-y") . '-' . time() . '.' . $extension;
        $path       = $file->storeAs('public/asset_bulk_update', $fileName);
        $path       = storage_path('app/' . $path);

        $errors = $this->specialBulkStatusUpdate->importAssetData($path, $fileName);

        if (isset($errors['validationErrors']) && !empty($errors['validationErrors'])) {
            Storage::put('public/asset_bulk_update/update-error-' . date("m-d-y") . '-' . time() . '.txt', json_encode($errors['validationErrors']));

            return redirect(route('bulk-status-update.index'))->with(['error' => implode(",\n", $errors['validationErrors'])]);
        }

        if ($errors && !isset($errors['validationErrors'])) {
            Storage::put('public/asset_bulk_update/update-error-' . date("m-d-y") . '-' . time() . '.txt', json_encode($errors));

            return redirect(route('bulk-status-update.index'))->with(['message' => "Bulk status updated with errors", 'errors' => $errors]);
        }

        return redirect(route('bulk-status-update.index'))->with(['message' => "Bulk status updated successfully"]);
    }
}
