<?php

namespace App\Http\Controllers\Assets\Settings;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\SpecialUpload;
use App\Services\BulkUpload\BulkMoveService;
use App\Services\BulkUpload\SpecialBulkMoveService;
use Illuminate\Support\Facades\Storage;
use App\Http\Requests\CsvFileUploadRequest;
use App\Http\Traits\FileSanitizationTrait;

class BulkMoveController extends Controller
{
    use FileSanitizationTrait;

    public function __construct(protected BulkMoveService $bulkMoveService, protected SpecialBulkMoveService $specialBulkMoveService) {}

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        session(['asset_count' => 0]);

        return view('settings.bulk-move');
    }

    /**
     * Handle the upload of a CSV file for bulk move asset updates.
     *
     * This method processes the uploaded file, stores it in a designated directory,
     * and delegates the file processing to the bulkMoveService.
     *
     * @param \App\Http\Requests\CsvFileUploadRequest $request
     *        The validated request containing the uploaded CSV file.
     *
     * @return mixed
     *         The response from the bulkMoveService after processing the file.
     */
    public function upload(CsvFileUploadRequest $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        // Get the uploaded file from the request
        $file = $request->file('file');

        // Get the file extension in lowercase
        $extension = strtolower($file->getClientOriginalExtension());

        $fileName = 'move-asset-items-' . date("m-d-y") . '-' . time() . '.' . $extension;
        $path = $file->storeAs('public/asset_bulk_move', $fileName);

        return $this->bulkMoveService->getViewData(storage_path('app/' . $path), $fileName);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $errors = [];
        $updatedCount = 0;

        if ($request->count == 0) {
            return redirect(route('bulk-move.index'))->with('error', 'Assets cannot be moved.');
        }

        $basePath = storage_path('app/public/asset_bulk_move');
        $filePath = $this->sanitizeFilePath($request->filePath, $basePath);
        $fileName = $this->sanitizeFileName($request->fileName);

        for ($i = 1; $i <= $request->count; $i++) {
            if (!$request->has('asset_id' . $i)) {
                continue; //row was deleted before submitting
            }

            $updateResponse = $this->bulkMoveService->saveData($request->all(), $i);

            if ($updateResponse === false) {
                $errors[] = request('serial_no' . $i);
                continue;
            }

            $updatedCount++;
        }

        $this->bulkMoveService->attachFiletoTicket(request('ticket_no1'), $filePath, $fileName);

        if (empty($errors)) {
            return redirect(route('bulk-move.index'))->with('message', 'Assets moved successfully.');
        }

        if ($request->count != count($errors)) {
            return redirect(route('bulk-move.index'))
                ->with([
                    'message' => "Total {$updatedCount} assets moved successfully.",
                    'error' => 'Assets ' . implode(', ', $errors) . ' cannot be moved.',
                ]);
        }

        return redirect(route('bulk-move.index'))->with('error', 'Assets cannot be moved.');
    }

    /**
     * Handles the special upload for bulk move asset processing.
     *
     * This function sets unlimited execution time and memory for handling large
     * file uploads. It stores the uploaded file in the public/asset_bulk_move
     * directory with a unique filename, and processes the file using the
     * specialBulkMove service to import asset data.
     *
     * Validation errors are logged in a timestamped file in the same directory,
     * and the function redirects back to the bulk-move.index route with the errors.
     * If other errors occur, they are also logged and the function redirects back
     * with a message indicating partial success along with the error details.
     *
     * If no errors occur, the function redirects back to the bulk-move.index route
     * with a success message.
     *
     * @param \App\Http\Requests\SpecialUpload $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function specialUpload(SpecialUpload $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $file   = $request->file('bulk_upload_asset');

        $extension  = strtolower($file->getClientOriginalExtension());
        $fileName   = 'move-asset-items-' . date("m-d-y") . '-' . time() . '.' . $extension;
        $path       = $file->storeAs('public/asset_bulk_move', $fileName);
        $path       = storage_path('app/' . $path);

        $errors = $this->specialBulkMoveService->importAssetData($path, $fileName);

        if (isset($errors['validationErrors']) && !empty($errors['validationErrors'])) {
            Storage::put('public/asset_bulk_move/move-error-' . date("m-d-y") . '-' . time() . '.txt', json_encode($errors['validationErrors']));

            return redirect(route('bulk-move.index'))->with(['error' => implode(",\n", $errors['validationErrors'])]);
        }

        if ($errors && !isset($errors['validationErrors'])) {
            Storage::put('public/asset_bulk_move/move-error-' . date("m-d-y") . '-' . time() . '.txt', json_encode($errors));
            $message = 'Bulk Move with errors';

            return redirect(route('bulk-move.index'))->with(['message' => $message, 'error' => json_encode($errors)]);
        }

        return redirect(route('bulk-move.index'))->with('message', 'Bulk move successful.');
    }
}
