<?php

namespace App\Http\Controllers\Assets\Settings;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\SpecialUpload;
use App\Services\BulkUpload\BulkAssignService;
use App\Services\BulkUpload\SpecialBulkAssignService;
use Illuminate\Support\Facades\Storage;
use App\Http\Requests\CsvFileUploadRequest;
use App\Http\Traits\FileSanitizationTrait;

class BulkAssignController extends Controller
{
    use FileSanitizationTrait;

    public function __construct(protected BulkAssignService $bulkStatusUpdate, protected SpecialBulkAssignService $specialBulkStatusUpdate) {}

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        session(['asset_count' => 0]);

        return view('settings.bulk-assign');
    }

    /**
     * Handle the upload of a CSV file for bulk asset assignment.
     *
     * This method sets unlimited execution time and memory limit for large file processing,
     * retrieves the uploaded file, saves it to a specific directory, and processes it
     * using the bulkStatusUpdate service.
     *
     * @param \Illuminate\Http\Request $request
     *        The HTTP request containing the uploaded file.
     *
     * @return mixed
     *         The response from the bulkStatusUpdate service after processing the file.
     */
    public function upload(CsvFileUploadRequest $request)
    {

        setUnlimitedExecutionTimeAndMemoryLimit();
        // Get the uploaded file from the request
        $file = $request->file('file');

        // Get the file extension in lowercase
        $extension = strtolower($file->getClientOriginalExtension());

        $fileName = 'assign-asset-items-' . date("m-d-y") . '-' . time() . '.' . $extension;

        $path = request()->file('file')->storeAs('public/asset_bulk_assign', $fileName);

        return $this->bulkStatusUpdate->getViewData(storage_path('app/' . $path), $fileName);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $errors = [];
        $updatedCount = 0;

        if ($request->count == 0) {
            return redirect(route('bulk-assign.index'))->with('error', 'Assets cannot be updated.');
        }

        $basePath = storage_path('app/public/asset_bulk_assign');
        $filePath = $this->sanitizeFilePath($request->filePath, $basePath);
        $fileName = $this->sanitizeFileName($request->fileName);

        for ($i = 1; $i <= $request->count; $i++) {

            if (!$request->has('asset_id' . $i)) {
                continue; //row was deleted before submitting
            }

            $updateResponse = $this->bulkStatusUpdate->saveData($request->all(), $i);
            if ($updateResponse === false) {
                $errors[] = request('serial_no' . $i);
                continue;
            }

            $updatedCount++;
        }

        $this->bulkStatusUpdate->attachFiletoTicket(request('ticket_no1'), $filePath, $fileName);
        if (empty($errors)) {

            return redirect(route('bulk-assign.index'))->with('message', 'Assets updated successfully.');
        }

        if ($request->count != count($errors)) {
            return redirect(route('bulk-assign.index'))
                ->with([
                    'message' => "Total {$updatedCount} assets updated successfully.",
                    'error' => 'Assets ' . implode(', ', $errors) . ' cannot be updated.',
                ]);
        }

        return redirect(route('bulk-assign.index'))->with('error', 'Assets cannot be updated.');
    }

    /**
     * Special upload function for bulk asset assignment.
     *
     * This function sets unlimited execution time and memory limit
     * to handle large file uploads. It stores the uploaded file in the
     * public/asset_bulk_assign directory with a unique filename.
     * The file is then processed by the specialBulkStatusUpdate service
     * to import asset data.
     *
     * If there are validation errors during import, it logs the errors
     * in a timestamped file in the public/asset_bulk_assign directory
     * and redirects back to the bulk-assign.index route with the errors.
     * If other errors occur, it logs them in a similar manner and redirects
     * with a message indicating partial success and detailing the errors.
     *
     * If no errors occur, it redirects back to the bulk-assign.index route
     * with a success message.
     *
     * @param \App\Http\Requests\SpecialUpload $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function specialUpload(SpecialUpload $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $file   = $request->file('bulk_upload_asset');

        $extension  = strtolower($file->getClientOriginalExtension());
        $fileName   = 'assign-asset-items-' . date("m-d-y") . '-' . time() . '.' . $extension;
        $path       = $file->storeAs('public/asset_bulk_assign', $fileName);
        $path       = storage_path('app/' . $path);

        $errors = $this->specialBulkStatusUpdate->importAssetData($path, $fileName);

        if (isset($errors['validationErrors']) && !empty($errors['validationErrors'])) {
            Storage::put('public/asset_bulk_assign/assign-error-' . date("m-d-y") . '-' . time() . '.txt', json_encode($errors['validationErrors']));

            return redirect(route('bulk-assign.index'))->with(['error' => implode(",\n", $errors['validationErrors'])]);
        }

        if ($errors && !isset($errors['validationErrors'])) {
            Storage::put('public/asset_bulk_assign/assign-error-' . date("m-d-y") . '-' . time() . '.txt', json_encode($errors));

            return redirect(route('bulk-assign.index'))->with(['message' => 'Bulk assign successful with errors', 'errors' => $errors]);
        }

        return redirect(route('bulk-assign.index'))->with('message', 'Bulk assign successful.');
    }
}
