<?php

namespace App\Http\Controllers\Assets\Settings;

use App\Events\BulkUpdates;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\AssetType;
use App\Models\AssetHistory;
use App\Models\AssetTab;
use Facades\App\Services\Settings\AssetType as AssetTypeService;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use App\Models\AssetHealthAssetType;

class AssetTypeController extends Controller
{

    public function __construct(AssetTypeService $assetTypeService)
    {
        $this->assetTypeService = $assetTypeService;
    }

    public function index()
    {
        $assetTypes = AssetType::withCount('assets')->orderBy('slug')->get();
        $healthTestAssetTypes = AssetHealthAssetType::pluck('type_id')->toArray();
        return view('settings.asset-type.asset-type', compact('assetTypes', 'healthTestAssetTypes'));
    }

    public function store()
    {
        $this->validate(request(), [
            'asset_type' => 'required',
        ]);

        $assetType = request('asset_type');
        $assetSlug = str_slug($assetType, '_');
        $asset = AssetType::where('slug', $assetSlug)->first();
        $assetTypeId = AssetTab::where('slug', 'it_assets')->first()->id;
        if (!$asset) {
            AssetType::create([
                'name' => $assetType,
                'slug' => str_slug($assetType, '_')
            ])->assetTabs()->attach($assetTypeId);

            $description = __('history.AssetTypeCreated', [
                'assetType' => $assetType,
            ]);
            $assetHistory = [
                'user_id' => Auth::id(),
                'action' => 'assetType_created',
                'description' => $description,
                'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            ];
            event(new BulkUpdates($assetHistory));

            return redirect('/asset-type');
        } else {
            return redirect('/asset-type')->withErrors(['Asset type already exists']);
        }
    }

    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }

    public function update()
    {
        $assetType = AssetType::findOrFail(request('asset_type_id'));
        $oldAssetType = $assetType->name;
        $assetType->update([
            'name' => request('name'),
            'slug' => str_slug(request('name'), '_')
        ]);

        $description = __('history.AssetTypeUpdated', [
            'oldAssetType' => $oldAssetType,
            'newAssetType' => request('name')
        ]);
        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'assetType_updated',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));

        return response()->json('success');
    }

    public function destroy()
    {
        $assetType = AssetType::findOrFail(request('asset_type_id'));
        $assetTypeValue = $assetType->name;
        if (count($assetType->assets)) {
            return response()->json('error');
        }
        $assetType->delete();

        $description = __('history.AssetTypeDeleted', [
            'assetType' => $assetTypeValue
        ]);
        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'assetType_deleted',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));
        return response()->json("Deleted Successfully");
    }

    /**
     * Asset Type Export
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        
        $data = [];
        $datas = collect();
        $assetTypes = AssetType::withCount('assets')->orderBy('slug');
        $assetTypes->chunk(5000, function ($assetTypeChunks) use ($data, $datas) {
            $datas->push($this->assetTypeService::getAssetTypeData($assetTypeChunks, 0, $data));
        });
        return $this->assetTypeService::export($datas->toArray());
    }
}
