<?php

namespace App\Http\Controllers\Assets;

use App\Http\Controllers\Controller;
use App\Models\AssetStatus;
use Facades\App\Services\Asset\AssetStatusService;
use App\Models\Asset;
use App\Services\Asset\SearchAssetService;
use Facades\App\Repositories\SoftwareLicense\LicenseRepository;

class SearchAssetController extends Controller
{

    public function __construct(protected SearchAssetService $searchAssetService) {}

    /**
     * Display the search results for IT assets.
     *
     * This method retrieves and displays the search results for IT assets based on
     * the search text and general type provided in the request. It gathers necessary
     * data such as asset statuses and determines the admin status of the user.
     * The results are then passed to the 'assets.search-asset' view for rendering.
     *
     * @return \Illuminate\View\View The view displaying the search results for IT assets.
     */

    public function index()
    {
        $breadcrumbTitle = 'IT Assets';
        $type = request('general_type');
        $searchText = request('searchText');

        $searchResults = $this->searchAssetService->getSearchedAssets($searchText);

        $statuses = AssetStatus::orderBy('slug')->get();
        request()->flash();

        $adminStatus = $this->searchAssetService->determineAdminStatus();

        return view('assets.search.index', array_merge($searchResults, compact(
            'statuses',
            'breadcrumbTitle',
            'type',
            'adminStatus'
        )));
    }

    /**
     * Retrieves search fields based on the provided search and type parameters.
     *
     * If the search results are empty, a message is added to the response indicating
     * where to search for the asset.
     *
     * @return \Illuminate\Http\JsonResponse The merged collection of search fields.
     */
    public function getSearchFields()
    {
        $search = request('search');
        $type = request('type');

        if (!empty($search)) {

            $result = $this->searchAssetService->getSearchFields($search, $type);

            return response()->json($result);
        }
    }

    /**
     * Gets the serial# and asset tags to populate the autocomplete input field
     * @return [type]
     */
    public function getSearchSerialAndAssetTagFields()
    {
        $search = request('search');
        $type = request('type');

        if (!empty($search)) {

            $result = $this->searchAssetService->getSearchSerialAndAssetTags($search, $type);

            return response()->json($result);
        }
    }

    /**
     * Get the total original value of assets based on search criteria.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getOriginalValueTotal(): \Illuminate\Http\JsonResponse
    {
        $result = ['sum' => 0];
        $user = null;

        $searchText = request('searchText', '');
        $type = request('type', '');

        return $this->searchAssetService->getOriginalValueTotal($searchText, $type, $user, $result);
    }

    /**
     * Get the total original value of assets in a location based on search criteria.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getOriginalValueLocationTotal()
    {
        $result = ['sum' => 0];
        $searchText = request('searchText'); // Store the searchText in a variable
        $type = request('type');
        $location = null;

        return $this->searchAssetService->getOriginalValueLocationTotal($searchText, $type, $location, $result);
    }

    /**
     * Get total value of software assets
     * @return [type]
     */
    public function getOriginalValueSoftwareTotal()
    {
        $result = ['sum' => 0];
        $searchedSoftwareAsset = LicenseRepository::assetUserSearched(request('user'))->get();

        $softwareAssetsValue = 0;
        $softwareAssetsValue = $searchedSoftwareAsset->map(function ($item) {
            return $item->license->cost;
        })->sum();

        $result = ['sum' => $softwareAssetsValue];

        return response()->json($result);
    }

    /**
     * Return the user ID for the given asset ID
     * @param int $id Asset ID
     * @return \Illuminate\Http\JsonResponse
     */
    public function getAssetUser()
    {
        request()->validate([
            'id' => 'required',
        ]);
        $assetId = request('id');
        $userid = Asset::find($assetId)->user_id;
        $result = ['id' => $userid];

        return response()->json($result);
    }

    /**
     * This function is used to list the asset statuses for the status update based on the asset current status
     */
    public function listAssetStatuses()
    {
        $assetStatuses = AssetStatusService::availableStatusesForUpdate(request('status'));

        return $assetStatuses->map(function ($item) {
            // Remove the current status
            if (request('status') != $item->slug) {
                return ['id' => $item->id, 'name' => $item->name];
            }
        });
    }

    /**
     * Updates the hardware and technical specs of an asset
     *
     * @return Redirect
     */
    public function updateHardwareTechspec()
    {
        $asset = Asset::find(request('asset_id'));
        $asset->make_and_model_id = request('make_and_model_id');
        $asset->technical_spec_id = request('technical_spec_id');
        $asset->save();

        return redirect()->back()->with('message', 'Asset Hardware / Tech Specs updated');
    }

    public function downloadEndOfFile($file)
    {
        // Sanitize file name (strictly allow only valid PDF filenames).
        if (!preg_match('/^[a-zA-Z0-9_\-\s\(\)]+\.pdf$/', $file)) {
            Log::warning("Invalid PDF file request: " . $file);
            abort(403, "Invalid file request.");
        }

        $headers = array(
            'Content-Type: application/pdf',
        );

        $file = str_replace("-", "/", $file);
        $filePath = storage_path('app/public/' . $file);


        // Check if the file exists
        if (file_exists($filePath)) {
            return response()->download($filePath, 'EndOfLife.pdf', $headers);
        } else {
            // File not found, return a error response
            return redirect()->back()->with('error', 'EOL File not found.');
        }
    }
}
