<?php

namespace App\Http\Controllers\Assets;

use App\Events\BulkUpdates;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\CsvFileUploadRequest;
use App\Models\AssetStatus;
use Facades\App\Services\AssetData;
use App\Models\Asset as AssetModel;
use Facades\App\Models\MakeAndModel;
use App\Models\AssetType;
use App\Http\Requests\StoreAssetRequest;
use App\Http\Responses\DataTableJsonResponse;
use Facades\App\Services\Asset\AssetStatusService;
use App\User;
use Facades\App\Services\JiraService;
use Illuminate\Support\Facades\Auth;
use App\Models\Location;
use Facades\App\Services\AssetBulkUpload;
use Cache;
use Facades\App\Repositories\AssetRepository;
use App\Models\UserType;
use Carbon\Carbon;

class ResearchAssetController extends Controller
{


    /**
     * Display a listing of asset statuses.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $statuses = AssetStatus::orderBy('slug')->get();

        return view('assets.asset-research', compact('statuses'));
    }


    /**
     * Retrieve and filter asset data for data tables.
     *
     * @param Request $request The incoming request instance.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the data for DataTables.
     */
    public function data(Request $request)
    {
        $filteredData   = AssetData::filter();
        $assets         = $filteredData['assets'];
        $totalData      = $filteredData['count'];

        $start  = request('start');
        $data   = [];

        if (!empty($assets)) {
            $data = AssetData::getResearchAssetData($assets, $start, $data);
        }

        return new DataTableJsonResponse($request->input('draw'), $data, $totalData);
    }


    /**
     * Show the form for creating a new research asset.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $assetTypes = AssetType::research();
        $data = ['title' => 'Create Research Asset', 'parent' => 'Asset', 'type' => 'research', 'parentUrl' => '/research-assets'];
        $hardwares = MakeAndModel::getAll();

        session(['count_add' => 0]);

        return view('assets.create-research', compact('hardwares', 'data', 'assetTypes'));
    }


    /**
     * Add a new asset and return the view with added assets.
     *
     * @param StoreAssetRequest $request The request instance containing asset data.
     * @param int        $count   The initial count of assets, defaults to 1.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the rendered view.
     */
    public function add(StoreAssetRequest $request, $count = 1)
    {
        if (session('count_add')) {
            $count = session('count_add') + 1;
        }

        session(['count_add' => $count]);
        $asset_status = AssetStatusService::whenCreatingAsset();

        $request['asset_status_id'] = AssetStatus::where('slug', $asset_status)->first()->id;
        $request['count_add'] = session('count_add');
        $request['asset_tag'] = AssetBulkUpload::getAssetTag($request['asset_tag']);

        $view = view('assets.partials.added-assets', compact('request'))->render();

        return response()->json($view);
    }


    /**
     * Store multiple assets and log the creation history.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store()
    {
        $count = request('count');

        $assetStatus = AssetStatusService::whenCreatingAsset();

        for ($i = 1; $i <= $count; $i++) {
            // Row was deleted before submitting.
            if (!request()->has('ticket_no' . $i)) {
                continue;
            }

            $data = $this->getData($i);
            $assetStatusId = AssetStatus::where('slug', $assetStatus)->first()->id;

            $data['asset_status_id'] = $assetStatusId;
            $location = Location::where('id', $data['location_id'])->with('locationType')->first();
            $asset_id = AssetModel::updateorCreate(['serial_no' => $data['serial_no']], $data)
                ->id;

            $user = User::find(Auth::id());

            $description = __('history.Created', [
                'assetname' => $data['serial_no'],
                'assetid' => $asset_id,
                'newroomname' => $location->room_name,
            ]);

            $jiraDescription = __('jira.Created', [
                'assetname' => $data['serial_no'],
                'assetid' => $asset_id,
                'username' => $user->first_name . ' ' . $user->last_name,
                'useremail' => $user->email,
            ]);

            $jiraData[] = [
                'description' => $jiraDescription,
                'ticketId'   => request('ticket_no' . $i),
                'userId' => $user->id,
            ];

            $assetHistory = [
                'user_id' => Auth::id(),
                'asset_id' => $asset_id,
                'ticket_no' => request('ticket_no' . $i),
                'ticket_service_provider' => config('ticket-integration.service'),
                'action' => 'created',
                'new_location_id' => $data['location_id'],
                'description' => $description,
                'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            ];

            event(new BulkUpdates($assetHistory));
        }

        session(['count_add' => 0]);

        JiraService::bulkSendToJira($jiraData);

        return redirect(route('research-assets.index'))->with('message', 'Assets created successfully.');
    }


    /**
     * Retrieve asset data from the request for a given count.
     *
     * @param int $count The index to retrieve specific asset data from the request.
     *
     * @return array The processed asset data.
     */
    public function getData($count)
    {
        $assetData = [];

        $assetData['po_id'] = request('po_id' . $count);
        $assetData['asset_tag'] = request('asset_tag' . $count);
        $assetData['location_id'] = request('location_id' . $count);
        $assetData['asset_type_id'] = request('asset_type_id' . $count);
        $assetData['make_and_model_id'] = request('make_and_model_id' . $count);
        $assetData['technical_spec_id'] = request('technical_spec_id' . $count);
        $assetData['serial_no'] = request('serial_no' . $count);
        $assetData['ticket_no'] = request('ticket_no' . $count);
        $assetData['ip'] = request('ip' . $count);
        $assetData['mac'] = request('mac' . $count);

        $serialNo = $assetData['serial_no'];

        if (($serialNo[0] == 'S') || ($serialNo[0] == 's')) {
            $assetData['serial_no'] = substr($serialNo, 1);
        }

        return $assetData;
    }


    /**
     * Display the specified asset details.
     *
     * @param int $id The ID of the asset to display.
     *
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $asset = AssetModel::with('location', 'user', 'assetType', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'parentAsset', 'carrier', 'assetHistory')->find($id);

        $hardwares = MakeAndModel::getAll();
        $data = ['title' => 'Research Asset Detail', 'parent' => 'Research Assets', 'type' => 'research', 'parentUrl' => '/research-assets'];

        return view('assets.detail', compact('asset', 'hardwares', 'data'));
    }

    /**
     * Handles the upload of a CSV file for research asset processing.
     *
     * This method processes the uploaded CSV file, stores it in a designated directory,
     * and imports the asset data using the AssetBulkUpload service. It validates the CSV
     * columns and research asset fields, rendering any errors found during validation. If there
     * are duplicate entries, it handles them accordingly. Finally, it returns a JSON response
     * containing the rendered views for errors, duplicate errors, and valid research asset data.
     *
     * @param \App\Http\Requests\CsvFileUploadRequest $request
     *        The validated request containing the uploaded CSV file.
     *
     * @return \Illuminate\Http\JsonResponse
     *         A JSON response containing the rendered errors or data view.
     */
    public function upload(CsvFileUploadRequest $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        // Get the uploaded file from the request
        $file = $request->file('file');

        // Get the file extension in lowercase
        $extension = strtolower($file->getClientOriginalExtension());

        $fileName = 'asset-items-' . date("m-d-y") . '-' . time() . '.' . $extension;
        $path = $file->storeAs('public/asset_bulk_upload', $fileName);

        $storagePath = storage_path('app/' . $path);

        $view = [];
        $count = session('count_add');
        $data = AssetBulkUpload::importAssetResearchData($storagePath, $count);

        if (!empty($data['error'])) {
            $errors = $data['error'];
            $view['errors'] = view('assets.partials.upload-csv-errors', compact('errors'))->render();
            return response()->json($view);
        }

        $csvData = $data['csvData'];
        $errors = AssetBulkUpload::validateCsvColumns($storagePath, $csvData, 'research');
        $view['errors'] = view('assets.partials.upload-csv-errors', compact('errors'))->render();

        if ($errors != '') {
            return response()->json($view);
        }

        if ($csvData) {
            $errors = AssetBulkUpload::validateResearchFields($csvData);
            $view['errors'] = view('assets.partials.upload-errors', compact('errors'))->render();

            if (!empty(array_filter($errors))) {
                return response()->json($view);
            }

            $duplicateErrors = AssetData::checkRepeatValues($csvData);
            $view['duplicate_errors'] = view('assets.partials.upload-duplicate-errors', compact('duplicateErrors'))->render();

            if (!empty(array_filter($errors)) || !empty($duplicateErrors)) {
                $csvData = [];
            }

            $view['data'] = view('assets.partials.upload-research-assets', compact('csvData'))->render();
            $view['countVal'] = count($csvData);
        }

        return response()->json($view);
    }

    /**
     * Search and display assets based on user input.
     *
     * @return \Illuminate\View\View
     */
    public function search()
    {
        $breadcrumbTitle = 'Manage Assets';
        $user = '';
        $location = '';
        $type = request('general_type');
        $searchedLocationAssetTagCount = 0;
        $searchedUserAssetTagCount = 0;

        $installLocations = $this->getCachedLocations('install-locations', 'withoutwarehouse');
        $warehouseLocations = $this->getCachedLocations('warehouse-locations', 'warehouses');
        $installAndBuildingLocations = $this->getCachedLocations('install-building-locations', 'installBuildings');

        if (request()->has('searchText') && !empty(request('searchText'))) {
            $searchText = request('searchText');

            $user = AssetRepository::searchAssetUser($searchText, 'research');
            $location = AssetRepository::searchAssetLocation($searchText, 'research');

            $searchedLocationAsset = $this->getSearchedAsset('assetLocationSearched', $searchText);
            $searchedUserAsset = $this->getSearchedAsset('assetUserSearched', $searchText);

            if ($searchedLocationAsset) {
                $searchedLocationAssetTagCount = 1;
            }

            if ($searchedUserAsset) {
                $searchedUserAssetTagCount = 1;
            }

            if ($location) {
                $location->assets = $this->paginateAssets($location->assets(), $searchText, $searchedLocationAsset);
            }

            if ($user) {
                $user->assets = $this->paginateAssets($user->assets(), $searchText, $searchedUserAsset);
            }
        }

        $statuses = AssetStatus::where('slug', '!=', 'assigned')->orderBy('slug')->get();
        request()->flash();

        $userTypes = ['Super User', 'Super Admin'];
        $userTypeName = UserType::find(User::find(Auth::id())->user_type_id)->name;
        $adminStatus = 0;

        if (in_array($userTypeName, $userTypes)) {
            $adminStatus = 1;
        }

        return view('assets.search-asset', compact('user', 'location', 'statuses', 'breadcrumbTitle', 'searchedUserAssetTagCount', 'searchedLocationAssetTagCount', 'type', 'installLocations', 'warehouseLocations', 'adminStatus', 'installAndBuildingLocations'));
    }


    /**
     * Retrieve the first searched asset based on the given search method and text.
     *
     * @param string $searchMethod The method to use for searching the asset (e.g., 'searchAssetUser', 'searchAssetLocation').
     * @param string $searchText   The text to search for in the asset.
     *
     * @return \App\Models\AssetModel|null The first matched asset with related data, or null if no asset is found.
     */
    private function getSearchedAsset($searchMethod, $searchText)
    {
        return AssetRepository::{$searchMethod}(
            AssetModel::researchAsset(),
            $searchText
        )->with(
            'user',
            'location',
            'assetType',
            'makeAndModel.manufacturer',
            'technicalSpec',
            'assetStatus',
            'carrier',
            'parentAsset',
            'childrenAsset'
        )->first();
    }


    /**
     * Paginate the assets and prepend the searched asset if applicable.
     *
     * @param \Illuminate\Database\Eloquent\Builder $assetsQuery   The query builder instance for the assets.
     * @param string                                $searchText    The text to search for in the assets.
     * @param \App\Models\AssetModel|null           $searchedAsset The searched asset to prepend, if any.
     *
     * @return \Illuminate\Pagination\LengthAwarePaginator The paginated assets with the searched asset prepended if applicable.
     */
    private function paginateAssets($assetsQuery, $searchText, $searchedAsset)
    {
        $assets = $assetsQuery->with(
            'user',
            'location',
            'assetType',
            'makeAndModel.manufacturer',
            'technicalSpec',
            'assetStatus',
            'carrier',
            'parentAsset',
            'childrenAsset'
        )->researchAsset()
            ->notSearched($searchText)
            ->paginate(50);

        if ($assets->currentPage() == 1 && $searchedAsset) {
            $assets->prepend($searchedAsset);
        }

        return $assets;
    }


    /**
     * Retrieve cached locations based on the given scope.
     *
     * @param string $cacheKey The key to use for caching the locations.
     * @param string $scope    The scope method to call on the Location model (e.g., 'withoutwarehouse', 'warehouses').
     *
     * @return \Illuminate\Support\Collection The collection of locations.
     */
    private function getCachedLocations($cacheKey, $scope)
    {
        return Cache::remember(
            $cacheKey,
            720,
            function () use ($scope) {
                return Location::{$scope}()->active()->orderBy('room_name', 'asc')->get();
            }
        );
    }
}
