<?php

namespace App\Http\Controllers\Assets;

use App\Events\BulkUpdates;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\CsvFileUploadRequest;
use App\Models\Asset as AssetModel;
use Facades\App\Models\MakeAndModel;
use Facades\App\Services\Asset\AssetStatusService;
use Facades\App\Services\JiraService;
use Facades\App\Services\AssetData;
use Facades\App\Services\AssetBulkUpload;
use App\Http\Requests\StoreAssetRequest;
use App\Http\Responses\DataTableJsonResponse;
use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Models\Location;
use App\Models\UserType;
use App\Models\Vendor;
use Illuminate\Support\Facades\Auth;
use App\User;
use Carbon\Carbon;
use Facades\App\Repositories\AssetRepository;
use Cache;

class ProdAssetController extends Controller
{

    public function index()
    {
        $statuses = AssetStatus::orderBy('slug')->get();

        return view('assets.asset-prod', compact('statuses'));
    }

    public function create()
    {
        $assetTypes = AssetType::avassets();
        $vendors = Vendor::orderBy('name')->get();
        $data = ['title' => 'Create AV Asset', 'parent' => 'Asset', 'type' => 'avassets', 'parentUrl' => route('av-assets-search')];

        return view('assets.create-prod-asset', compact('data', 'assetTypes', 'vendors'));
    }

    public function store()
    {
        $count = request('count');

        for ($i = 1; $i <= $count; $i++) {

            if (!request()->has('ticket_no' . $i)) continue; //row was deleted before submitting

            $data = $this->getData($i);
            $assetStatusId = AssetStatus::where('slug', 'installed')->first()->id;

            $location = Location::where('id', $data['location_id'])->with('locationType')->first();

            if ($location->locationType->location_type_name == 'Install') {
                $assetStatusId = AssetStatus::where('slug', 'installed')->first()->id;
            }

            $data['asset_status_id'] = $assetStatusId;

            $asset_id = AssetModel::updateorCreate(['serial_no' => $data['serial_no']], $data)
                ->id;

            $user = User::find(Auth::id());
            $description = __('history.Created', [
                'assetname' => $data['serial_no'],
                'assetid' => $asset_id,
                'newroomname' => $location->room_name,
            ]);

            if ($location->locationType->location_type_name == 'Install') {
                $description = __('history.CreatedAndInstalled', [
                    'assetname' => $data['serial_no'],
                    'assetid' => $asset_id,
                    'room_name' => $location->room_name,
                ]);
            }

            $jiraDescription = __('jira.Created', [
                'assetname' => $data['serial_no'],
                'assetid' => $asset_id,
                'username' => $user->first_name . ' ' . $user->last_name,
                'useremail' => $user->email,
            ]);

            $jiraData[] = [
                'description' => $jiraDescription,
                'ticketId'   => request('ticket_no' . $i),
                'userId' => $user->id,
            ];

            $assetHistory = [
                'user_id' => Auth::id(),
                'asset_id' => $asset_id,
                'ticket_no' => request('ticket_no' . $i),
                'ticket_service_provider' => config('ticket-integration.service'),
                'action' => 'created',
                'new_location_id' => $data['location_id'],
                'description' => $description,
                'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            ];

            event(new BulkUpdates($assetHistory));
        }

        session(['count_add' => 0]);

        JiraService::bulkSendToJira($jiraData);

        return redirect(route('av-assets.index'))->with('message', 'AV assets created successfully.');
    }


    public function show($id)
    {
        $asset = AssetModel::with('location', 'user', 'assetType', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'parentAsset', 'carrier', 'assetHistory')->find($id);
        $hardwares = MakeAndModel::getAll();

        $data = ['title' => 'AV Asset Detail', 'parent' => 'AV Assets', 'type' => 'avassets', 'parentUrl' => '/av-assets'];

        return view('assets.detail', compact('asset', 'hardwares', 'data'));
    }

    public function add(StoreAssetRequest $request, $count = 1)
    {
        if (session('count_add')) {
            $count = session('count_add') + 1;
        }

        session(['count_add' => $count]);
        $asset_status = AssetStatusService::whenCreatingAsset();
        $request['asset_status_id'] = AssetStatus::where('slug', $asset_status)->first()->id;
        $request['count_add'] = session('count_add');
        $request['asset_tag'] = AssetBulkUpload::getAssetTag($request['asset_tag']);

        $view = view('assets.partials.added-av-assets', compact('request'))->render();

        return response()->json($view);
    }

    public function getData($count)
    {
        $assetData['location_id'] = request('location_id' . $count);
        $assetData['asset_type_id'] = request('asset_type_id' . $count);
        $assetData['asset_tag'] = request('asset_tag' . $count);
        $assetData['po_id'] = request('po_id' . $count);
        $assetData['technical_spec_id'] = request('technical_spec_id' . $count);
        $assetData['serial_no'] = request('serial_no' . $count);
        $assetData['ticket_no'] = request('ticket_no' . $count);
        $assetData['make_and_model_id'] = request('make_and_model_id' . $count);
        $assetData['asset_status_id'] = request('asset_status_id' . $count);
        $assetData['carrier_id'] = request('carrier_id' . $count);
        $assetData['imei'] = request('imei' . $count);
        $assetData['ip'] = request('ip' . $count);
        $assetData['mac'] = request('mac' . $count);

        if (request('technical_spec_id' . $count) == "--Select--") {
            $assetData['technical_spec_id'] = null;
        }
        if (request('make_and_model_id' . $count) == "--Select--") {
            $assetData['make_and_model_id'] = null;
        }
        if (request('imei' . $count) == "") {
            $assetData['imei'] = null;
        }

        if (in_array(request('asset_type_id' . $count), AssetType::leased()->pluck('id')->toArray())) {
            $assetData['lease_start_date'] = parse_date_from_db_datetime(Carbon::now());
            $assetData['lease_end_date'] = parse_date_from_db_datetime(Carbon::now()->addYears(config('date.period.leased')));
        }

        $serialNo = $assetData['serial_no'];

        if (($serialNo[0] == 'S') || ($serialNo[0] == 's')) {
            $assetData['serial_no'] = substr($serialNo, 1);
        }

        return $assetData;
    }

    /**
     * Handle the upload of a CSV file for AV asset processing.
     *
     * This method processes the uploaded CSV file, stores it in a designated directory,
     * and imports the asset data using the AssetBulkUpload service. It validates the CSV
     * columns and AV asset fields, rendering any errors found during validation. If there
     * are duplicate entries, it handles them accordingly. Finally, it returns a JSON response
     * containing the rendered views for errors, duplicate errors, and valid AV asset data.
     *
     * @param \App\Http\Requests\CsvFileUploadRequest $request
     *        The validated request containing the uploaded CSV file.
     *
     * @return \Illuminate\Http\JsonResponse
     *         A JSON response containing the rendered errors or data view.
     */
    public function upload(CsvFileUploadRequest $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        // Get the uploaded file from the request
        $file = $request->file('file');

        // Get the file extension in lowercase
        $extension = strtolower($file->getClientOriginalExtension());

        $fileName = 'av-asset-items-' . date("m-d-y") . '-' . time() . '.' . $extension;
        $path = $file->storeAs('public/asset_bulk_upload', $fileName);

        $path = storage_path('app/' . $path);

        $view = [];
        $count = session('count_add');

        $data = AssetBulkUpload::importAvAssetData($path, $count);
        if (!empty($data['error'])) {
            $errors = $data['error'];
            $view['errors'] = view('assets.partials.upload-csv-errors', compact('errors'))->render();
            return response()->json($view);
        }

        $csvData = $data['csvData'];


        $errors = AssetBulkUpload::validateCsvColumns($path, $csvData, 'network');
        $view['errors'] = view('assets.partials.upload-csv-errors', compact('errors'))->render();

        if ($errors != '') {
            return response()->json($view);
        }

        if ($csvData) {

            $errors = AssetBulkUpload::validateAvFields($csvData);
            $view['errors'] = view('assets.partials.upload-errors', compact('errors'))->render();

            if (!empty(array_filter($errors))) {
                return response()->json($view);
            }

            $duplicateErrors = AssetData::checkRepeatNetworkValues($csvData);
            $view['duplicate_errors'] = view('assets.partials.upload-duplicate-errors', compact('duplicateErrors'))->render();

            if (!empty(array_filter($errors)) || !empty($duplicateErrors)) {
                $csvData = [];
            }

            $view['data'] = view('assets.partials.upload-av-assets', compact('csvData'))->render();
            $view['countVal'] = count($csvData);
        }

        return response()->json($view);
    }

    public function data(Request $request)
    {
        $filteredData   = AssetData::filter();
        $assets         = $filteredData['assets'];
        $totalData      = $filteredData['count'];

        $start  = request('start');
        $data   = [];

        if (!empty($assets)) {
            $data = AssetData::getServerAssetData($assets, $start, $data);
        }

        return new DataTableJsonResponse($request->input('draw'), $data, $totalData);
    }

    public function search()
    {
        $breadcrumbTitle = 'Manage Assets';
        $user = '';
        $location = '';
        $type = request('general_type');
        $searchedLocationAssetTagCount = 0;
        $searchedUserAssetTagCount = 0;
        $installLocations = Cache::remember('install-locations', 720, function () {
            return Location::withoutwarehouse()->active()->orderBy('room_name', 'asc')->get();
        });
        $warehouseLocations = Cache::remember('warehouse-locations', 720, function () {
            return Location::warehouses()->active()->orderBy('room_name', 'asc')->get();
        });

        $installAndBuildingLocations = Cache::remember('install-building-locations', 720, function () {
            return Location::installBuildings()->active()->orderBy('room_name', 'asc')->get();
        });

        if (request()->has('searchText') && !empty(request('searchText'))) {
            $user = AssetRepository::searchAssetUser(request('searchText'), 'av');
            $location = AssetRepository::searchAssetLocation(request('searchText'), 'av');

            $query = AssetModel::avAsset();
            $searchedLocationAsset = AssetRepository::assetLocationSearched($query, request('searchText'))->with('user', 'location', 'assetType', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'carrier', 'parentAsset', 'childrenAsset')->first();


            $query = AssetModel::avAsset();
            $searchedUserAsset = AssetRepository::assetUserSearched($query, request('searchText'))->with('user', 'location', 'assetType', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'carrier', 'parentAsset', 'childrenAsset')->first();

            if ($searchedLocationAsset) {
                $searchedLocationAssetTagCount = 1;
            }

            if ($searchedUserAsset) {
                $searchedUserAssetTagCount = 1;
            }

            if ($location) {
                $location->assets = $location->assets()->with('user', 'location', 'assetType', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'carrier', 'parentAsset', 'childrenAsset')->avAsset()->notSearched(request('searchText'));
                $location->assets = $location->assets->paginate(50);

                if ($location->assets->currentPage() == 1 && $searchedLocationAsset) {
                    $location->assets->prepend($searchedLocationAsset);
                }
            }

            if ($user) {
                $user->assets = $user->assets()->with('user', 'location', 'assetType', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'carrier', 'parentAsset', 'childrenAsset')->avAsset()->notSearched(request('searchText'));
                $user->assets = $user->assets->paginate(50);

                if ($user->assets->currentPage() == 1 && $searchedUserAsset) {
                    $user->assets->prepend($searchedUserAsset);
                }
            }
        }
        $statuses = AssetStatus::where('slug', '!=', 'assigned')->orderBy('slug')->get();
        request()->flash();

        $userTypes = ['Super User', 'Super Admin'];
        $userTypeName = UserType::find(User::find(Auth::id())->user_type_id)->name;
        $adminStatus = 0;

        if (in_array($userTypeName, $userTypes)) {
            $adminStatus = 1;
        }

        return view('assets.search-asset', compact('user', 'location', 'statuses', 'breadcrumbTitle', 'searchedUserAssetTagCount', 'searchedLocationAssetTagCount', 'type', 'installLocations', 'warehouseLocations', 'installAndBuildingLocations', 'adminStatus'));
    }
}
