<?php

namespace App\Http\Controllers\Assets;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\CsvFileUploadRequest;
use App\Http\Requests\StoreAssetRequest;
use App\Models\AssetType;
use App\Models\AssetTypeAttribute;
use App\Models\Vendor;
use App\Services\Asset\AssetStatusService;
use App\Services\Asset\CreateAssetService;
use App\Services\Asset\AssetBulkCreateService;
use App\Services\Integrations\Tickets\TicketManagementService;

class CreateController extends Controller
{

    public function __construct(
        protected CreateAssetService $createAssetService,
        protected AssetStatusService $assetStatusService,
        protected TicketManagementService $ticketManagementService,
        protected AssetBulkCreateService $assetBulkUploadService
    ) {}


    /**
     * Displays the form for creating a new IT asset.
     * 
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $assetTypes = AssetType::regular();
        $data = ['title' => 'Create IT Asset', 'parent' => 'Asset', 'type' => 'it_assets', 'parentUrl' => '/assets'];
        $vendors = Vendor::get();
        session(['count_add' => 0]);
        return view('assets.create.index', compact('data', 'assetTypes', 'vendors'));
    }

    /**
     * Displays the form for creating a new mobile asset.
     * 
     * @return \Illuminate\Http\Response
     */
    public function mobile()
    {
        $assetTypes = AssetType::mobile();
        $data = ['title' => 'Create Mobile Asset', 'parent' => 'Asset', 'type' => 'mobile_assets', 'parentUrl' => route('assets.mobile')];
        $vendors = Vendor::get();
        session(['count_add' => 0]);
        return view('assets.create.index', compact('data', 'assetTypes', 'vendors'));
    }

    /**
     * Add a new asset and return the view with added assets.
     *
     * @param StoreAssetRequest $request The request instance containing asset data.
     * @param int        $count   The initial count of assets, defaults to 1.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the rendered view.
     */
    public function add(StoreAssetRequest $request, $count = 1)
    {
        if (session('count_add')) {
            $count = session('count_add') + 1;
        }

        session(['count_add' => $count]);

        $assetStatus = $this->assetStatusService->getStatusForNewAsset();
        $request['asset_status_id'] = $assetStatus?->id;
        $request['count_add'] = session('count_add');
        $request['asset_tag'] = $this->createAssetService->getAssetTag($request['asset_tag']);

        $view = view('assets.partials.added-assets', compact('request'))->render();

        return response()->json($view);
    }

    /**
     * Handle the creation of a new asset or multiple assets.
     * 
     * This function validates the request for at least one asset, and then
     * calls the CreateAssetService to create the new asset(s).
     * 
     * It then redirects to the corresponding index route based on the
     * category of the asset, and displays a success message.
     * 
     * If all assets are not created, it displays an error message.
     * 
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'count' => 'required|gt:0'
        ], ['count.required' => 'Please add some assets to create']);

        $createdCount = $this->createAssetService->createNewAsset($request->all());

        $category = $request->category;

        // Determine the base route for redirection based on category
        // Determine the route and message based on category using switch
        switch ($category) {
            case 'mobile':
                $route = route('assets.mobile');
                $message = ucfirst($category) . " assets created successfully.";
                break;
            case 'research':
                $route = url('research-assets');
                $message = "Research assets created successfully.";
                break;
            default:
                $route = route('assets.index');
                $message = "{$createdCount} assets created successfully.";
                break;
        }

        // If all assets are not created, add an error message
        if ($request->count != $createdCount) {
            return redirect($route)->with([
                'message' => "Total {$createdCount} assets created successfully.",
                'error' => 'Some assets could not be created.',
            ]);
        }

        // Success message for all created assets
        return redirect($route)->with('message', $message);
    }

    /**
     * Handles the advanced IT asset upload process.
     *
     * This method sets unlimited execution time and memory limit, stores the uploaded CSV file
     * in a designated directory, and retrieves the count from the session. It then returns
     * the view data for IT assets using the asset bulk upload service.
     *
     * @param \App\Http\Requests\CsvFileUploadRequest $request
     *        The validated request containing the uploaded CSV file.
     *
     * @return mixed
     *         The view data for IT assets.
     */
    public function advancedItUpload(CsvFileUploadRequest $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        // Get the uploaded file from the request
        $file = $request->file('file');

        // Get the file extension in lowercase
        $extension = strtolower($file->getClientOriginalExtension());

        $fileName = 'asset-items-' . date("m-d-y") . '-' . time() . '.' . $extension;
        $path = $file->storeAs('public/asset_bulk_upload', $fileName);

        $path = storage_path('app/' . $path);

        $count = session('count_add');

        return $this->assetBulkUploadService->getViewData($path, $count, 'it_assets');
    }

    /**
     * Handles the advanced mobile asset upload process.
     *
     * This method sets unlimited execution time and memory limit, stores the uploaded
     * file in a designated directory, and retrieves the count from the session. It then
     * returns the view data for mobile assets using the asset bulk upload service.
     *
     * @param \App\Http\Requests\CsvFileUploadRequest $request
     *        The validated request containing the uploaded file.
     *
     * @return mixed
     *         The view data for mobile assets.
     */
    public function advancedMobileUpload(CsvFileUploadRequest $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $file   = $request->file('file');

        $extension  = strtolower($file->getClientOriginalExtension());
        $fileName   = 'asset-items-mobile-' . date("m-d-y") . '-' . time() . '.' . $extension;
        $path       = $file->storeAs('public/asset_bulk_upload', $fileName);

        $path = storage_path('app/' . $path);

        $count = session('count_add');

        return $this->assetBulkUploadService->getViewData($path, $count, 'mobile_assets');
    }

    /**
     * Handles the bulk upload of assets process.
     *
     * This method sets unlimited execution time and memory limit, stores the uploaded
     * CSV file in a designated directory, and retrieves the count from the session.
     * It then returns the view data for the bulk upload using the asset bulk upload
     * service.
     *
     * @param \App\Http\Requests\CsvFileUploadRequest $request
     *        The validated request containing the uploaded CSV file.
     *
     * @return mixed
     *         The view data for the bulk upload.
     */
    public function bulkUpload(CsvFileUploadRequest $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $uploadData = $this->createAssetService->getData(request()->all());
        // remove asset_tag and serial_no since these are parsed from uploaded file
        unset($uploadData['asset_tag'], $uploadData['serial_no']);

        $uploadData['type'] = request('type');

        // Get the uploaded file from the request
        $file = $request->file('file');

        // Get the file extension in lowercase
        $extension = strtolower($file->getClientOriginalExtension());

        $fileName = 'asset-items-' . date("m-d-y") . '-' . time() . '.' . $extension;
        $path = $file->storeAs('public/asset_bulk_upload', $fileName);

        $path = storage_path('app/' . $path);

        $count = session('count_add');

        return $this->assetBulkUploadService->getBulkUploadViewData($path, $uploadData, $count);
    }

    /**
     * Gets the extra attributes for a given asset type
     *
     * Retrieves the extra attributes for the given asset type and returns the view
     * for the extra attributes.
     *
     * @return \Illuminate\Http\JsonResponse The response containing the view data.
     */
    public function listAdditionalAttributes()
    {
        $attributes = AssetTypeAttribute::GetAttributesByAssetType(request('id'))->get();
        $attributesView = view('assets.partials.extra-attributes', compact('attributes'))->render();

        return response()->json($attributesView);
    }
}
