<?php

namespace App\Http\Controllers\Asn\Shi;

use App\Http\Requests\Asn\Shi\StoreShiHardwareMapping;
use Facades\App\Repositories\HardwareStandards;
use App\Services\Asn\Shi\ShiHardwareMappingService;
use App\Http\Controllers\Controller;
use Facades\App\Models\MakeAndModel;
use App\Models\AsnHardwareMapping;
use App\Models\TechnicalSpecs;
use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Models\Asset;

class ShiHardwareMappingController extends Controller
{

    /**
     * @var ShiHardwareMappingService
     */
    protected $hardwareMappingService;


    /**
     * HardwareMappingController constructor.
     *
     * @param ShiHardwareMappingService $hardwareMappingService
     *
     */
    public function __construct(ShiHardwareMappingService $hardwareMappingService)
    {
        $this->hardwareMappingService = $hardwareMappingService;
    }


    /**
     * Display a listing of SHI hardware mappings.
     *
     * @return \Illuminate\View\View The view displaying the SHI hardware mappings.
     */
    public function index()
    {
        $asnMappings        = $this->hardwareMappingService->getAsnMappingData();
        $hardwareStandards  = HardwareStandards::getHardwareStandardsList();
        $hardwareStandardsArray = HardwareStandards::getHardwareStandardsList()->pluck('makeAndModel', 'id');
        $assettypes         = AssetType::pluck('name', 'id');
        $techSpechs         = TechnicalSpecs::get()->pluck('details', 'id');

        return view('asn.shi.shi-hardware-mapping', compact('asnMappings', 'assettypes', 'hardwareStandards', 'hardwareStandardsArray', 'techSpechs'));
    }


    /**
     * Search for SHI hardware mappings.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the rendered search results view.
     */
    public function search()
    {
        $asnMappings        = $this->hardwareMappingService->filter();
        $view['hardware']   = view('asn.shi.partials.data.shi-hardware-mapping-data', compact('asnMappings'))->render();

        return response()->json($view);
    }


    /**
     * Store a new SHI hardware mapping.
     *
     * @param \App\Http\Requests\StoreShiHardwareMapping $request The incoming HTTP request validated using the StoreShiHardwareMapping request class.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function store(StoreShiHardwareMapping $request)
    {
        $venderMapping = AsnHardwareMapping::where(['provider' => 'shi', 'part_no' => $request->description])
            ->first();

        if (!$venderMapping) {
            AsnHardwareMapping::Create([
                'provider' => 'shi',
                'make_and_model_id' => request('make_and_model'),
                'technical_spec_id' => request('technical_spec'),
                'part_no'           => request('description'),

            ]);

            if (request('sync_now') == 1) {
                $assetStatus = AssetStatus::whereIn('slug', ['shi_in_transit'])->pluck('id');
                $this->hardwareMappingService->individualHardwarePartResync($assetStatus, request('description'), request('make_and_model'), request('technical_spec'));
            }

            return redirect('/shi-hardware-mapping')->with('message', "Hardware Mapped Successfully");
        }

        return redirect('/shi-hardware-mapping')->withErrors(['Mapping for this description already exist']);
    }


    /**
     * Update an existing SHI hardware mapping.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message.
     */
    public function update()
    {
        AsnHardwareMapping::findOrFail(request('id'))
            ->update([
                'make_and_model_id' => request('make_and_model'),
                'technical_spec_id' => request('technical_spec'),
                'part_no'           => request('description'),
            ]);

        return response()->json(['status' => 'success', 'message' => "Hardware mapping updated successfully."]);
    }


    /**
     * Delete an existing SHI hardware mapping.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and success message.
     */
    public function destroy()
    {
        $asnHardwareMapping = AsnHardwareMapping::findOrFail(request('id'));
        $assetStatus = AssetStatus::whereIn('slug', ['shi_in_transit'])->pluck('id');
        $this->hardwareMappingService->resyncHardwarePartOnDelete($assetStatus, $asnHardwareMapping);
        $asnHardwareMapping->delete();

        return response()->json(['status' => 'success', 'message' => "Hardware mapping deleted successfully."]);
    }


    /**
     * Handle AJAX requests for updating or deleting SHI hardware mappings.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message from the update or destroy method, or a boolean value.
     */
    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }


    /**
     * Handle the bulk upload of SHI hardware mappings.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function bulkUpload()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $result = $this->hardwareMappingService->bulkUpload();

        return redirect('/shi-hardware-mapping')->with(['message' => $result['message'] ?? '', 'error' => $result['errors'] ?? '']);
    }


    /**
     * Resync SHI hardware mappings.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success message.
     */
    public function resync()
    {
        $allAsnHardwareMappings = AsnHardwareMapping::where('provider', 'shi')->whereNotNull('part_no')->get();
        $assetStatus = AssetStatus::where('slug', 'shi_in_transit')->first()->id;

        foreach ($allAsnHardwareMappings as $asnHardwareMapping) {
            $makeAndModel = MakeAndModel::findOrFail($asnHardwareMapping->make_and_model_id);

            Asset::where('asset_status_id', $assetStatus)
                ->where('part_no', $asnHardwareMapping->part_no)
                ->update([
                    'make_and_model_id' => $asnHardwareMapping->make_and_model_id,
                    'technical_spec_id' => $asnHardwareMapping->technical_spec_id,
                    'asset_type_id'     => $makeAndModel->asset_type_id
                ]);
        }

        return redirect('/shi-hardware-mapping')->with('message', "Resynced Successfully");
    }
}
