<?php

namespace App\Http\Controllers\Asn\Shi;

use App\Services\Asn\Shi\ShiAssetsService;
use App\Http\Controllers\Controller;
use App\Models\TechnicalSpecs;
use App\Models\AssetHistory;
use Illuminate\Http\Request;
use App\Models\AssetStatus;
use App\Models\Asset;
use App\Models\AssetType;
use Exception;
use App\Http\Responses\DataTableJsonResponse;
use App\Http\Traits\Asn\AsnAssetTrait;
use App\Models\IntegrationSyncHistory;
use Facades\App\Repositories\HardwareStandards;
use Illuminate\Support\Facades\Artisan;

/**
 * Controller class for the ASN-SHI Assets management
 */
class ShiAssetsController extends Controller
{
    use AsnAssetTrait;

    /**
     * @var ShiAssetsService
     */
    protected $assetService;


    /**
     * Constructor for the controller.
     *
     * @param \App\Services\ShiAssetsService $assetService The assets service instance.
     */
    public function __construct(ShiAssetsService $assetService)
    {
        $this->assetService = $assetService;
    }


    /**
     * Display a listing of SHI assets.
     *
     * @return \Illuminate\View\View The view displaying the SHI assets.
     */
    public function index()
    {
        $hardwareStandards  = HardwareStandards::getHardwareStandardsWithoutAccessories();
        $hardwareStandardsArray = HardwareStandards::getHardwareStandardsWithoutAccessories()->pluck('makeAndModel', 'id');
        $assettypes         = AssetType::allWithoutComputerAccessories()->pluck('name', 'id');
        $techSpechs         = TechnicalSpecs::get()->pluck('details', 'id');
        $lastSync = IntegrationSyncHistory::getLastSync(config('integration-sync-histories.asn-shi'))->first();
        $lastSyncDate = $lastSync?->created_at;

        return view('asn.shi.shi-assets', compact('assettypes', 'hardwareStandards', 'hardwareStandardsArray', 'techSpechs', 'lastSyncDate'));
    }


    /**
     * Retrieve filtered data for SHI assets.
     *
     * @param \Illuminate\Http\Request $request The incoming HTTP request containing DataTable parameters.
     *
     * @return \App\Http\Responses\DataTableJsonResponse The JSON response containing the filtered asset data.
     */
    public function data(Request $request)
    {
        $filteredData  = $this->assetService->data('shi_in_transit');
        $assets        = $filteredData['assets'];

        $start = request('start');
        $data = [];

        if (!empty($assets)) {
            $data = $this->assetService->getAsnAssetData($assets, $start, $data);
        }

        return new DataTableJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }


    /**
     * Export SHI assets data to CSV.
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse The CSV file response.
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $data = [];
        $datas = collect();

        $filteredData = $this->assetService->getExportData('shi_in_transit')->get();
        $datas->push($this->assetService->getAsnAssetExportData($filteredData, 0, $data, 'SHI'));

        return exportToCsv($datas->toArray());
    }


    /**
     * Update an existing SHI asset.
     *
     * @return \Illuminate\Http\JsonResponse|string The JSON response with validation errors or a success message.
     */
    public function updateAsset()
    {
        $error = $this->assetService->validateAsset();

        if ($error != false) {
            return response()->json($error);
        }

        $asset = Asset::findOrFail(request('id'));

        if ($asset) {
            $data = $this->getDataToUpdate();
            $asset->update($data);
        }

        return "success";
    }


    /**
     * Delete an existing SHI asset.
     *
     * @param int $id The ID of the asset to be deleted.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with a success or error message.
     */
    public function deleteAsset($id)
    {
        try {
            $asset = Asset::select('id')
                ->whereIn('asset_status_id', AssetStatus::where('slug', ['shi_in_transit'])->pluck('id'))
                ->findOrFail($id);

            AssetHistory::where('asset_id', $id)->delete();
            $asset->delete();

            return response()->json('success');
        } catch (Exception $e) {
            return response()->json('error');
        }
    }


    /**
     * Mark assets as received.
     *
     * @return mixed The result of the `receiveAsnAsset` method, which handles the asset status update.
     */
    public function markReceived()
    {
        $oldStatus = AssetStatus::where('slug', 'shi_in_transit')->first();

        return $this->recieveAsnAsset($oldStatus);
    }


    /**
     * Manually sync SHI assets.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the output of the Artisan command.
     */
    public function manualSync()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        Artisan::call('ShiImport:import');

        return response()->json(Artisan::output());
    }
}
