<?php

namespace App\Http\Controllers\Asn\Shi;

use App\Http\Requests\Asn\Shi\StoreShiAccessoriesMapping;
use App\Services\Asn\Shi\ShiAccessoriesMappingService;
use Facades\App\Repositories\HardwareStandards;
use App\Models\AsnAccessoriesMapping;
use App\Http\Controllers\Controller;
use App\Models\AssetHistory;
use App\Models\AssetStatus;
use App\Models\Asset;

class ShiAccessoriesMappingController extends Controller
{

    /**
     * @var ShiAccessoriesMappingService
     */
    protected $accessoriesMappingService;


    /**
     * Constructor for the controller.
     *
     * @param \App\Services\ShiAccessoriesMappingService $accessoriesMappingService The accessories mapping service instance.
     */
    public function __construct(ShiAccessoriesMappingService $accessoriesMappingService)
    {
        $this->accessoriesMappingService = $accessoriesMappingService;
    }


    /**
     * Display a listing of SHI accessories mappings.
     *
     * @return \Illuminate\View\View The view displaying the SHI accessories mappings.
     */
    public function index()
    {
        $accessoriesMappings = $this->accessoriesMappingService->getAccessoriesMappingData();
        $hardwareStandards = HardwareStandards::getHardwareStandardsAccessories();
        $hardwareStandardsArray = HardwareStandards::getHardwareStandardsAccessories()->pluck('makeAndModel', 'id');

        return view('asn.shi.shi-accessories-mapping', compact('accessoriesMappings', 'hardwareStandards', 'hardwareStandardsArray'));
    }


    /**
     * Search for SHI accessories mappings.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the rendered search results view.
     */
    public function search()
    {
        $accessoriesMappings = $this->accessoriesMappingService->getAccessoriesMappingSearch();

        $view['hardware'] = view('asn.shi.partials.data.shi-accessories-mapping-data', compact('accessoriesMappings'))->render();

        return response()->json($view);
    }


    /**
     * Store a new SHI accessories mapping.
     *
     * @param \App\Http\Requests\StoreShiAccessoriesMapping $request The incoming HTTP request validated using the StoreShiAccessoriesMapping request class.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function store(StoreShiAccessoriesMapping $request)
    {
        $accessoriesMapping = AsnAccessoriesMapping::where(['provider' => 'shi', 'part_no' => $request->description])->first();

        if (!$accessoriesMapping) {
            AsnAccessoriesMapping::Create([
                'part_no'       => request('description'),
                'provider' => 'shi',
                'make_and_model_id' => request('make_and_model'),
            ]);

            return redirect('/shi-accessories-mapping')->with('message', 'Accessories mapping added successfully.');
        }

        return redirect('/shi-accessories-mapping')->withErrors(['Mapping already exist']);
    }


    /**
     * Update an existing SHI accessories mapping.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message.
     */
    public function update()
    {
        AsnAccessoriesMapping::findOrFail(request('id'))
            ->update([
                'part_no'       => request('description'),
                'make_and_model_id' => request('make_and_model'),
            ]);

        return response()->json(['status' => 'success', 'message' => "Accessory mapping updated successfully."]);
    }


    /**
     * Delete an existing SHI accessories mapping.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and success message.
     */
    public function destroy()
    {
        $accessoriesMapping = AsnAccessoriesMapping::findOrFail(request('id'));
        $accessoriesMapping->delete();

        return response()->json(['status' => 'success', 'message' => "Accessory mapping deleted successfully."]);
    }


    /**
     * Handle AJAX requests for updating or deleting SHI accessories mappings.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message from the update or destroy method, or a boolean value.
     */
    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }


    /**
     * Handle the bulk upload of SHI accessories mappings.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function bulkUpload()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $result = $this->accessoriesMappingService->bulkUpload();

        if (isset($result['errors'])) {
            return redirect('/shi-accessories-mapping')->with(['message' => $result['message'] ?? '', 'error' => $result['errors'] ?? '']);
        }

        return redirect('/shi-accessories-mapping')->with(['message' => $result['message'] ?? '']);
    }


    /**
     * Resync SHI accessories mappings.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success message.
     */
    public function resync()
    {
        $allAccessoriesMappings = AsnAccessoriesMapping::where('provider', 'shi')->whereNotNull('part_no')->get();
        $status                 = AssetStatus::whereIn('slug', ['shi_in_transit'])->first();

        foreach ($allAccessoriesMappings as $accessoriesMapping) {
            $acessories = Asset::select('id')
                ->where('asset_status_id', $status->id)
                ->where('part_no', $accessoriesMapping->part_no)
                ->get();

            foreach ($acessories as $accessory) {
                AssetHistory::where('asset_id', $accessory->id)->delete();
                $accessory->delete();
            }
        }

        return redirect('/shi-accessories-mapping')->with('message', "Resynced Successfully");
    }
}
