<?php

namespace App\Http\Controllers\Asn\Presidio;

use App\Models\AssetType;
use App\Models\MakeAndModel;
use App\Models\TechnicalSpecs;
use App\Models\AsnHardwareMapping;
use App\Http\Controllers\Controller;
use Facades\App\Repositories\HardwareStandards;
use App\Http\Requests\Asn\Presidio\StoreAsnHardwareMapping;
use App\Http\Requests\CsvFileUploadRequest;
use App\Services\Asn\Presidio\PresidioHardwareMappingService;

class PresidioHardwareMappingController extends Controller
{
    /**
     * HardwareMappingController constructor.
     *
     * @param PresidioHardwareMappingService $service
     *
     */
    public function __construct(protected PresidioHardwareMappingService $service)
    {
        $this->service = $service;
    }

    /**
     * Display a listing of Presidio hardware mappings.
     *
     * @return \Illuminate\View\View The view displaying the Presidio hardware mappings.
     */
    public function index()
    {
        $presidioHardwareMappings   = $this->service->getPresidioMappingData();
        $hardwareStandards          = HardwareStandards::getHardwareStandardsWithoutAccessories();
        $hardwareStandardsArray     = $hardwareStandards->pluck('makeAndModel', 'id');
        $assetTypes                 = AssetType::pluck('name', 'id');
        $techSpecs                  = TechnicalSpecs::pluck('details', 'id');

        return view('asn.presidio.hardware-mapping', compact('presidioHardwareMappings', 'assetTypes', 'hardwareStandards', 'hardwareStandardsArray', 'techSpecs'));
    }

    /**
     * Search for Presidio hardware mappings.
     *
     * Retrieves filtered hardware mappings data and returns a JSON response
     * containing the rendered search results view.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with rendered search results.
     */
    public function search()
    {
        $presidioHardwareMappings   = $this->service->filter();

        $view['hardware'] = view('asn.presidio.partials.data.hardware-mapping', compact('presidioHardwareMappings'))->render();

        return response()->json($view);
    }

    /**
     * Store a new Presidio hardware mapping.
     *
     * If the mapping already exists, redirects back with an error message.
     *
     * @param StoreAsnHardwareMapping $request The incoming HTTP request validated using the StoreAsnHardwareMapping request class.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function store(StoreAsnHardwareMapping $request)
    {
        $presidioHardwareMapping = AsnHardwareMapping::where('provider', 'presidio')->where('part_no', $request->part_no)->exists();

        if ($presidioHardwareMapping) {

            return redirect()->back()->withErrors(['Mapping for this part # already exist']);
        }

        $this->service->store();

        return redirect()->back()->with('message', "Hardware mapped successfully");
    }

    /**
     * Handle AJAX requests for updating or deleting Presidio hardware mappings.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message from the update or destroy method, or a boolean value.
     */

    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        }

        if (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }

    /**
     * Update specific Mapping and update hardware details of related assets
     *
     * @return response
     */
    public function update()
    {
        $techSpec           = TechnicalSpecs::findOrFail(request('technical_spec'));
        $makeModel          = MakeAndModel::findOrFail(request('make_and_model'));
        $hardwareMapping    = AsnHardwareMapping::findOrFail(request('id'));

        if ($hardwareMapping) {

            $hardwareMapping->update([
                'make_and_model_id' => $makeModel ? $makeModel->id : null,
                'technical_spec_id' => $techSpec ? $techSpec->id : null,
                'part_no'           => request('part_no'),
            ]);

            return response()->json(['status' => 'success', 'message' => 'Selected hardware mapping updated successfully.']);
        }

        return response()->json(['status' => 'error', 'message' => 'Failed to update selected hardware mapping.']);
    }

    /**
     * Delete specific Mapping
     *
     * @return response
     */
    public function destroy()
    {
        $presidioHardwareMapping = AsnHardwareMapping::findOrFail(request('id'));

        if ($presidioHardwareMapping) {

            $presidioHardwareMapping->delete();
            return response()->json(['status' => 'success', 'message' => 'Selected hardware mapping deleted successfully.']);
        }

        return response()->json(['status' => 'error', 'message' => 'Failed to delete selected hardware mapping.']);
    }

    /**
     * Handles the bulk upload of Presidio hardware mappings.
     *
     * @param CsvFileUploadRequest $request
     *
     * Sets unlimited execution time and memory limit for the operation.
     * Calls the bulkUpload service method to process the upload.
     * Redirects to the Presidio hardware mapping page with a success or error message.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function bulkUpload(CsvFileUploadRequest $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        $result = $this->service->bulkUpload($request);

        if (isset($result['errors'])) {
            session()->flash('error', $result['errors']);
        }

        session()->flash('message', $result['message'] ?? '');

        return response()->json(['status' => true]);
    }


    /**
     * Resyncs Presidio hardware mappings.
     *
     * Retrieves all Presidio hardware mappings with associated make and model details,
     * and updates the corresponding Presidio assets using the mapping information.
     * Redirects to the Presidio hardware mapping page with a success message upon completion.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success message.
     */
    public function resync()
    {
        $allPresidioHardwareMappings = AsnHardwareMapping::with('makeAndModel.assetType')->where('provider', 'presidio')->whereNotNull('part_no')->get();

        foreach ($allPresidioHardwareMappings as $key => $presidioHardwareMapping) {

            $this->service->updatePresidioAssets(
                $presidioHardwareMapping->part_no,
                $presidioHardwareMapping->make_and_model_id,
                $presidioHardwareMapping->technical_spec_id,
                $presidioHardwareMapping->makeAndModel->asset_type_id
            );
        }

        return redirect('/presidio-hardware-mapping')->with('message', "Resynced successfully");
    }
}
