<?php

namespace App\Http\Controllers\Asn\Presidio;

use App\Models\Asset;
use App\Models\AssetStatus;
use App\Models\AssetHistory;
use App\Http\Controllers\Controller;
use App\Models\AsnAccessoriesMapping;
use Illuminate\Support\Facades\Cache;
use App\Http\Traits\Asn\AsnAssetTrait;
use Facades\App\Repositories\HardwareStandards;
use App\Services\Asn\Presidio\AccessoriesMappingService;
use App\Http\Requests\Asn\Presidio\StoreAsnAccessoriesMapping;
use App\Http\Requests\CsvFileUploadRequest;

class PresidioAccessoriesMappingController extends Controller
{
    use AsnAssetTrait;

    /**
     * Constructs a new instance of the class.
     *
     * @param AccessoriesMappingService $service The AccessoriesMappingService object.
     */
    public function __construct(protected AccessoriesMappingService $service)
    {
        $this->service = $service;
    }

    /**
     * Display a listing of Presidio accessories mappings.
     *
     * @return \Illuminate\View\View The view displaying the Presidio accessories mappings.
     */
    public function index()
    {
        $presidioAccessoriesMappings    = $this->service->getAccessoriesMappingData();
        $hardwareStandards              = HardwareStandards::getHardwareStandardsAccessories();
        $hardwareStandardsArray         = $hardwareStandards->pluck('makeAndModel', 'id');
        return view('asn.presidio.accessories-mapping', compact('presidioAccessoriesMappings', 'hardwareStandards', 'hardwareStandardsArray'));
    }

    /**
     * Search for Presidio accessories mappings.
     *
     * Retrieves filtered accessories mappings data and returns a JSON response
     * containing the rendered search results view.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with rendered search results.
     */
    public function search()
    {
        $presidioAccessoriesMappings    = $this->service->getAccessoriesMappingSearch();
        $view['hardware']               = view('asn.presidio.partials.data.asn-accessories-mapping-data', compact('presidioAccessoriesMappings'))->render();

        return response()->json($view);
    }

    /**
     * Stores a new Presidio accessories mapping.
     *
     * Checks if the mapping already exists before creating a new one.
     * If the mapping already exists, redirects back with error message.
     *
     * @param StoreAsnAccessoriesMapping $request The incoming HTTP request validated using the StoreAsnAccessoriesMapping request class.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function store(StoreAsnAccessoriesMapping $request)
    {
        $accessoriesMapping = AsnAccessoriesMapping::where('provider', 'presidio')->where('part_no', $request->part_no)->exists();

        if ($accessoriesMapping) {
            return redirect('/presidio-accessories-mapping')->withErrors(['Mapping already exist']);
        }

        AsnAccessoriesMapping::create([
            'provider'          => 'presidio',
            'part_no'           => $request->part_no,
            'make_and_model_id' => $request->make_and_model,
        ]);

        return redirect('/presidio-accessories-mapping')->with('message', "Accessory mapping added successfully");
    }

    /**
     * Updates an existing Presidio accessories mapping.
     *
     * Retrieves the accessories mapping with the given ID and updates it with the
     * provided part number and make and model ID. If the mapping does not exist,
     * returns a JSON error response.
     *
     * Also removes any assets that have the same description as the updated part
     * number and have the status of "Presidio In Transit".
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message.
     */
    public function update()
    {
        $presidioInTransitId        = $this->getPresidioInTransitId();
        $presidioAccessoriesMapping = AsnAccessoriesMapping::findOrFail(request('id'));

        $presidioAccessoriesMapping->update(['part_no' => request('part_no'), 'make_and_model_id' => request('make_and_model')]);
        $acessories = Asset::select('id')->where('asset_status_id', $presidioInTransitId)->where('description', $presidioAccessoriesMapping->part_no)->get();

        foreach ($acessories as $accessory) {
            AssetHistory::where('asset_id', $accessory->id)->delete();
            $accessory->delete();
        }

        return response()->json(['status' => 'success', 'message' => 'Updated successfully']);
    }

    /**
     * Deletes an existing Presidio accessories mapping.
     *
     * Retrieves the accessories mapping with the given ID and deletes it.
     * If the mapping does not exist, returns a JSON error response.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message.
     */
    public function destroy()
    {
        $presidioAccessoriesMapping = AsnAccessoriesMapping::findOrFail(request('id'));

        if ($presidioAccessoriesMapping) {
            $presidioAccessoriesMapping->delete();

            return response()->json(['status' => 'success', 'message' => 'Deleted Successfully']);
        }

        return response()->json(['status' => 'error', 'message' => 'Failed to delete.']);
    }

    /**
     * Handle AJAX requests for updating or deleting Presidio accessories mappings.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message from the update or destroy method, or a boolean value.
     */
    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        }

        if (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }

    /**
     * Handles the bulk upload of Presidio accessories mappings.
     *
     * @param CsvFileUploadRequest $request
     *
     * Sets unlimited execution time and memory limit for the operation.
     * Calls the bulkUpload service method to process the upload.
     * Redirects to the Presidio accessories mapping page with a success or error message.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function bulkUpload(CsvFileUploadRequest $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $result = $this->service->bulkUpload($request);

        if (isset($result['errors'])) {
            session()->flash('error', $result['errors']);
        }

        session()->flash('message', $result['message'] ?? '');

        return response()->json(['status' => true]);
    }

    /**
     * Resyncs Presidio accessories mappings.
     *
     * Retrieves all Presidio accessories mappings and the ID of the 'presidio_in_transit' asset status.
     * If there are any mappings, removes any unwanted parts by calling the removeUnWantedParts method.
     * Redirects to the Presidio accessories mapping page with a success message.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success message.
     */
    public function resync()
    {
        $allPresidioAccessoriesMapping  = AsnAccessoriesMapping::where('provider', 'presidio')->whereNotNull('part_no')->get();
        $presidioInTransitId            = $this->getPresidioInTransitId();

        if (!empty($allPresidioAccessoriesMapping)) {
            $this->removeUnWantedParts($allPresidioAccessoriesMapping, $presidioInTransitId);
        }

        return redirect('/presidio-accessories-mapping')->with('message', "Resynced Successfully");
    }

    /**
     * Gets the ID of the 'presidio_in_transit' asset status from the asset_statuses table with caching.
     *
     * @return int The ID of the 'presidio_in_transit' asset status.
     */
    public function getPresidioInTransitId()
    {
        return Cache::remember('asset_status:presidio_in_transit_id', 3600, function () {
            return AssetStatus::where('slug', 'presidio_in_transit')->value('id');
        });
    }
}
