<?php

namespace App\Http\Controllers\Asn\Insight;

use App\Http\Controllers\Controller;
use App\Models\TechnicalSpecs;
use App\Models\AsnHardwareMapping;
use Facades\App\Models\MakeAndModel;
use App\Models\AssetType;
use App\Models\Asset;
use App\Models\AssetStatus;
use Facades\App\Repositories\HardwareStandards;
use App\Services\Asn\Insight\InsightHardwareMappingService;
use App\Http\Requests\Asn\Insight\StoreInsightHardwareMapping;

class InsightHardwareMappingController extends Controller
{

    /**
     * @var InsightHardwareMappingService
     */
    protected $hardwareMappingService;

    /**
     * HardwareMappingController constructor.
     *
     * @param InsightHardwareMappingService $hardwareMappingService
     *
     */
    public function __construct(InsightHardwareMappingService $hardwareMappingService)
    {
        $this->hardwareMappingService = $hardwareMappingService;
    }


    /**
     * Display a listing of insight hardware mappings.
     *
     * @return \Illuminate\View\View The view displaying the insight hardware mappings.
     */
    public function index()
    {
        $insightMappings            = $this->hardwareMappingService->getInsightMappingData();
        $hardwareStandards      = HardwareStandards::getHardwareStandardsWithoutAccessories();
        $hardwareStandardsArray = HardwareStandards::getHardwareStandardsWithoutAccessories()->pluck('makeAndModel', 'id');
        $assettypes             = AssetType::allWithoutComputerAccessories()->pluck('name', 'id');
        $techSpechs             = TechnicalSpecs::get()->pluck('details', 'id');

        return view('asn.insight.insight-hardware-mapping', compact('insightMappings', 'assettypes', 'hardwareStandards', 'hardwareStandardsArray', 'techSpechs'));
    }


    /**
     * Search for insight hardware mappings.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the rendered search results view.
     */
    public function search()
    {
        $insightMappings      = $this->hardwareMappingService->filter();
        $view['hardware'] = view('asn.insight.partials.data.insight-hardware-mapping-data', compact('insightMappings'))->render();

        return response()->json($view);
    }


    /**
     * Store a new insight hardware mapping.
     *
     * @param \App\Http\Requests\StoreInsightHardwareMapping $request The incoming HTTP request validated using the StoreInsightHardwareMapping request class.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function store(StoreInsightHardwareMapping $request)
    {
        $venderMappingExist = AsnHardwareMapping::where(['provider' => 'insight', 'part_no' => $request->description])->first();

        if (!$venderMappingExist) {
            $this->hardwareMappingService->store();

            return redirect('/insight-hardware-mapping')->with('message', "Hardware mapped successfully.");
        }

        return redirect('/insight-hardware-mapping')->withErrors(['Mapping for this description already exist']);
    }


    /**
     * Handle AJAX requests for updating or deleting insight hardware mappings.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message from the update or destroy method, or a boolean value.
     */
    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }


    /**
     * Update an existing insight hardware mapping.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message.
     */
    public function update()
    {
        $techSpec  = TechnicalSpecs::where('id', request('technical_spec'))->first();
        $makeModel = MakeAndModel::where('id', request('make_and_model'))->first();

        AsnHardwareMapping::findOrFail(request('id'))
            ->update([
                'make_and_model_id' => $makeModel ? $makeModel->id : null,
                'technical_spec_id' => $techSpec ? $techSpec->id : null,
                'part_no'       => request('description'),
            ]);

        return response()->json(['status' => 'success', 'message' => "Hardware mapping updated successfully."]);
    }


    /**
     * Delete an existing insight hardware mapping.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and success message.
     */
    public function destroy()
    {
        $asnHardwareMapping = AsnHardwareMapping::findOrFail(request('id'));
        $assetStatus = AssetStatus::whereIn('slug', ['insight_in_transit'])->pluck('id');

        $this->hardwareMappingService->resyncHardwarePartOnDelete($assetStatus, $asnHardwareMapping);
        $asnHardwareMapping->delete();

        return response()->json(['status' => 'success', 'message' => "Hardware mapping deleted successfully."]);
    }


    /**
     * Handle the bulk upload of insight hardware mappings.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function bulkUpload()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $result = $this->hardwareMappingService->bulkUpload();

        return redirect('/insight-hardware-mapping')->with(['message' => $result['message'] ?? '', 'error' => $result['errors'] ?? '']);
    }


    /**
     * Resync insight hardware mappings.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success message.
     */
    public function resync()
    {
        $allInsightHardwareMappings = AsnHardwareMapping::where('provider', 'insight')->whereNotNull('part_no')->get();

        foreach ($allInsightHardwareMappings as $insightHardwareMapping) {
            $makeAndModel  = MakeAndModel::findOrFail($insightHardwareMapping->make_and_model_id);
            $assetStatusId = AssetStatus::whereIn('slug', ['insight_in_transit'])->pluck('id');

            Asset::whereIn('asset_status_id', $assetStatusId)
                ->where('part_no', $insightHardwareMapping->part_no)
                ->update([
                    'make_and_model_id' => $insightHardwareMapping->make_and_model_id,
                    'technical_spec_id' => $insightHardwareMapping->technical_spec_id,
                    'asset_type_id'     => $makeAndModel->asset_type_id
                ]);
        }

        return redirect('/insight-hardware-mapping')->with('message', "Resynced Successfully.");
    }
}
