<?php

namespace App\Http\Controllers\Asn\Insight;

use App\Http\Controllers\Controller;
use App\Http\Responses\DataTableJsonResponse;
use App\Models\TechnicalSpecs;
use Facades\App\Models\Asset;
use App\Models\AssetStatus;
use App\Models\AssetType;
use Facades\App\Repositories\HardwareStandards;
use Illuminate\Http\Request;
use App\Models\AssetHistory;
use App\Services\Asn\Insight\InsightAssetsService;
use App\Http\Traits\Asn\AsnAssetTrait;
use App\Models\IntegrationSyncHistory;
use Illuminate\Support\Facades\Artisan;

class InsightAssetsController extends Controller
{
    use AsnAssetTrait;

    /**
     * @var InsightAssetsService
     */
    protected $assetService;


    /**
     * Constructor for InsightAssetsController
     *
     * @param InsightAssetsService $assetService
     */
    public function __construct(InsightAssetsService $assetService)
    {
        $this->assetService = $assetService;
    }


    /**
     * Display a listing of insight receive assets.
     *
     * @return \Illuminate\View\View The view displaying the insight receive assets.
     */
    public function index()
    {
        $hardwareStandards      = HardwareStandards::getHardwareStandardsWithoutAccessories();
        $hardwareStandardsArray = HardwareStandards::getHardwareStandardsWithoutAccessories()->pluck('makeAndModel', 'id');
        $techSpechs        = TechnicalSpecs::get()->pluck('details', 'id');
        $assettypes        = AssetType::allWithoutComputerAccessories()->pluck('name', 'id');
        $lastSync = IntegrationSyncHistory::getLastSync(config('integration-sync-histories.asn-insight'))->first();
        $lastSyncDate = $lastSync?->created_at;

        return view('asn.insight.insight-receive-assets', compact('hardwareStandards', 'hardwareStandardsArray', 'assettypes', 'techSpechs', 'lastSyncDate'));
    }


    /**
     * Handle AJAX requests for updating or deleting insight receive assets.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message from the update or destroy method, or a boolean value.
     */
    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }


    /**
     * Mark an asset as received.
     *
     * @return mixed The result of the `receiveAsnAsset` method, which handles the asset status update.
     */
    public function markReceived()
    {
        $oldStatus = AssetStatus::where('slug', 'insight_in_transit')->first();

        return $this->recieveAsnAsset($oldStatus);
    }


    /**
     * Retrieve filtered data for insight assets.
     *
     * @param \Illuminate\Http\Request $request The incoming HTTP request containing DataTable parameters.
     *
     * @return \App\Http\Responses\DataTableJsonResponse The JSON response containing the filtered asset data.
     */
    public function data(Request $request)
    {
        $filteredData  = $this->assetService->data('insight_in_transit');
        $assets        = $filteredData['assets'];

        $start = request('start');
        $data = [];

        if (!empty($assets)) {
            $data = $this->assetService->getAsnAssetData($assets, $start, $data);
        }

        return new DataTableJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }


    /**
     * Update an existing insight asset.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with validation errors or a success message.
     */
    public function updateAsset()
    {
        $error = $this->assetService->validateAsset();

        if ($error != false) {
            return response()->json($error);
        }

        $asset = Asset::findOrFail(request('id'));

        if ($asset) {
            $data = $this->getDataToUpdate();
            $asset->update($data);
        }
        return response()->json("success");
    }


    /**
     * Delete an existing insight asset.
     *
     * @param int $id The ID of the asset to be deleted.
     *
     * @return string A success message indicating the deletion status.
     */
    public function deleteAsset($id)
    {
        $assetStatusId = AssetStatus::where('slug', 'insight_in_transit')->pluck('id');
        $asset         = Asset::where('asset_status_id', $assetStatusId)->find($id);

        if (!$asset) {
            return "success";
        }

        $asset->delete();
        AssetHistory::where('asset_id', $id)->delete();

        return "success";
    }


    /**
     * Export insight assets data to CSV.
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse The CSV file response.
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $data = [];
        $datas = collect();

        $filteredData = $this->assetService->getExportData('insight_in_transit')->get();
        $datas->push($this->assetService->getAsnAssetExportData($filteredData, 0, $data, 'Insight'));

        return exportToCsv($datas->toArray());
    }


    /**
     * Synchronize insight assets.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the status and message of the synchronization.
     */
    public function sync()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        Artisan::call('insight:upload');
        $message = Artisan::output();

        if (substr($message, 0, 37) == "Insight assets creation completed, on") {
            return response()->json(['status' => 'success', 'message' => $message]);
        } else if (substr($message, 0, 45) == "Insight assets creation completed with errors") {
            $errors = view('asn.insight.partials.insight-sync-error-list')->render();

            return response()->json(['status' => 'error', 'message' => $message, 'errors' => $errors]);
        } else {
            return response()->json(['status' => 'error', 'message' => $message]);
        }
    }
}
