<?php

namespace App\Http\Controllers\Asn\Insight;

use App\Http\Controllers\Controller;
use App\Models\Asset;
use App\Models\AssetStatus;
use App\Models\AsnAccessoriesMapping;
use App\Services\Asn\Insight\AccessoriesMappingService;
use Facades\App\Repositories\HardwareStandards;
use App\Http\Requests\Asn\Insight\StoreInsightAccessoriesMapping;
use App\Models\AssetHistory;

class InsightAccessoriesMappingController extends Controller
{

    /**
     * @var AccessoriesMappingService
     */
    protected $accessoriesMappingService;


    /**
     * AccessoriesMappingController Constructor
     *
     * @param AccessoriesMappingService $accessoriesMappingService
     */
    public function __construct(AccessoriesMappingService $accessoriesMappingService)
    {
        $this->accessoriesMappingService = $accessoriesMappingService;
    }


    /**
     * Display a listing of insight accessories mappings.
     *
     * @return \Illuminate\View\View The view displaying the insight accessories mappings.
     */
    public function index()
    {
        $accessoriesMappings    = $this->accessoriesMappingService->getAccessoriesMappingData();
        $hardwareStandards      = HardwareStandards::getHardwareStandardsAccessories();
        $hardwareStandardsArray = HardwareStandards::getHardwareStandardsAccessories()->pluck('makeAndModel', 'id');

        return view('asn.insight.insight-accessories-mapping', compact('accessoriesMappings', 'hardwareStandards', 'hardwareStandardsArray'));
    }


    /**
     * Search for insight accessories mappings.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the rendered search results view.
     */
    public function search()
    {
        $accessoriesMappings = $this->accessoriesMappingService->getAccessoriesMappingSearch();

        $view['hardware'] = view('asn.insight.partials.data.insight-accessories-mapping-data', compact('accessoriesMappings'))->render();

        return response()->json($view);
    }


    /**
     * Store a new insight accessories mapping.
     *
     * @param \App\Http\Requests\StoreInsightAccessoriesMapping $request The incoming HTTP request validated using the StoreInsightAccessoriesMapping request class.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function store(StoreInsightAccessoriesMapping $request)
    {
        $accessoriesMapping = AsnAccessoriesMapping::where(['provider' => 'insight', 'part_no' => $request->description])->first();

        if ($accessoriesMapping) {
            return redirect('/insight-accessories-mapping')->withErrors(['Mapping already exist']);
        }

        AsnAccessoriesMapping::Create([
            'part_no' => request('description'),
            'provider' => 'insight',
            'make_and_model_id' => request('make_and_model'),
        ]);

        return redirect('/insight-accessories-mapping');
    }


    /**
     * Handle AJAX requests for updating or deleting insight accessories mappings.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message from the update or destroy method, or a boolean value.
     */
    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }


    /**
     * Update an existing insight accessories mapping.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message.
     */
    public function update()
    {
        AsnAccessoriesMapping::findOrFail(request('id'))
            ->update([
                'part_no' => request('description'),
                'make_and_model_id' => request('make_and_model'),
            ]);

        return response()->json(['status' => 'success', 'message' => "Accessory mapping updated successfully."]);
    }


    /**
     * Delete an existing insight accessories mapping.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and success message.
     */
    public function destroy()
    {
        AsnAccessoriesMapping::findOrFail(request('id'))->delete();

        return response()->json(['status' => 'success', 'message' => "Accessory mapping deleted successfully."]);
    }


    /**
     * Handle the bulk upload of insight accessories mappings.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function bulkUpload()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $result = $this->accessoriesMappingService->bulkUpload();

        if (isset($result['errors'])) {
            return redirect('/insight-accessories-mapping')->with(['message' => $result['message'] ?? '', 'error' => $result['errors'] ?? '']);
        }

        return redirect('/insight-accessories-mapping')->with(['message' => $result['message'] ?? '']);
    }


    /**
     * Resync insight accessories mappings.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success message.
     */
    public function resync()
    {
        $assetStatusId = AssetStatus::where('slug', 'insight_in_transit')->pluck('id');
        $allMappings   = AsnAccessoriesMapping::where('provider', 'insight')->whereNotNull('part_no')->get();

        foreach ($allMappings as $accessoriesMapping) {
            $acessories = Asset::where('asset_status_id', $assetStatusId)
                ->where('part_no', $accessoriesMapping->description)
                ->get();

            foreach ($acessories as $accessory) {
                AssetHistory::where('asset_id', $accessory->id)->delete();
                $accessory->delete();
            }
        }

        return redirect('/insight-accessories-mapping')->with('message', "Resynced successfully.");
    }
}
