<?php

namespace App\Http\Controllers\Asn\Connection;

use App\Http\Controllers\Controller;
use App\Models\AsnUnwantedPart;
use App\Models\Asset;
use App\Models\AssetStatus;
use App\Services\Asn\Connection\ConnectionUnwantedPartsService;
use Illuminate\Http\Request;
use App\Http\Requests\Asn\Connection\StoreConnectionUnwantedParts;
use App\Models\AssetHistory;

class ConnectionUnwantedPartsController extends Controller
{

    /**
     * @var \App\Services\Asn\Connection\ConnectionUnwantedPartsService
     */
    protected $unwantedPartsService;


    /**
     * Constructor for the controller.
     *
     * @param \App\Services\ConnectionUnwantedPartsService $unwantedPartsService The unwanted parts service instance.
     */
    public function __construct(ConnectionUnwantedPartsService $unwantedPartsService)
    {
        $this->unwantedPartsService = $unwantedPartsService;
    }


    /**
     * Display a listing of connection unwanted parts.
     *
     * @return \Illuminate\View\View The view displaying the unwanted parts.
     */
    public function index()
    {
        $unwantedParts = $this->unwantedPartsService->getUnwantedMappingData();

        return view('asn.connection.connection-unwanted-parts', compact('unwantedParts'));
    }


    /**
     * Search for connection unwanted parts.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the rendered search results view.
     */
    public function search()
    {
        $unwantedParts = $this->unwantedPartsService->getUnwantedMappingSearch();

        $view['hardware'] = view('asn.connection.partials.data.connection-unwanted-parts-data', compact('unwantedParts'))->render();

        return response()->json($view);
    }


    /**
     * Store a new connection unwanted parts mapping.
     *
     * @param \App\Http\Requests\StoreConnectionUnwantedParts $request The incoming HTTP request validated using the StoreConnectionUnwantedParts request class.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function store(StoreConnectionUnwantedParts $request)
    {
        $unwantedPartsMapping = AsnUnwantedPart::where(['provider' => 'connection', 'part_no' => $request->description])
            ->first();

        if (!$unwantedPartsMapping) {
            AsnUnwantedPart::Create([
                'part_no' => request('description'),
                'provider' => 'connection'
            ]);

            return redirect('/connection-unwanted-parts')->with('message', 'Mapping added successfully.');
        }

        return redirect('/connection-unwanted-parts')->withErrors(['Mapping already exist']);
    }


    /**
     * Update an existing connection unwanted parts mapping.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message.
     */
    public function update()
    {
        AsnUnwantedPart::findOrFail(request('id'))
            ->update([
                'part_no' => request('description')
            ]);

        return response()->json(['status' => 'success', 'message' => "Unwanted parts mapping updated successfully."]);
    }


    /**
     * Delete an existing connection unwanted parts mapping.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and success message.
     */
    public function destroy()
    {
        $unwantedPartsMapping = AsnUnwantedPart::findOrFail(request('id'));
        $unwantedPartsMapping->delete();

        return response()->json(['status' => 'success', 'message' => "Unwanted parts mapping deleted successfully."]);
    }


    /**
     * Handle AJAX requests for updating or deleting connection unwanted parts mappings.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message from the update or destroy method, or a boolean value.
     */
    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }


    /**
     * Handle the bulk upload of connection unwanted parts mappings.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function bulkUpload()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $result = $this->unwantedPartsService->bulkUpload();

        if (isset($result['errors'])) {
            return redirect('/connection-unwanted-parts')->with(['message' => $result['message'] ?? '', 'error' => $result['errors'] ?? '']);
        }

        return redirect('/connection-unwanted-parts')->with(['message' => $result['message'] ?? '']);
    }


    /**
     * Resync connection unwanted parts mappings.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success message.
     */
    public function resync()
    {
        $allConnectionUnwantedMapping = AsnUnwantedPart::where('provider', 'connection')->whereNotNull('part_no')->get();
        $connectionAssetStatus = AssetStatus::whereIn('slug', ['connection_in_transit'])->first();

        foreach ($allConnectionUnwantedMapping as $unwantedMapping) {
            $unwantedAssets = Asset::where('asset_status_id', $connectionAssetStatus->id)->where('part_no', $unwantedMapping->description)
                ->get();

            foreach ($unwantedAssets as $unwantedAsset) {
                AssetHistory::where('asset_id', $unwantedAsset->id)->delete();
                $unwantedAsset->delete();
            }
        }

        return redirect('/connection-unwanted-parts')->with('message', "Resynced Successfully");
    }
}
