<?php

namespace App\Http\Controllers\Asn\Connection;

use App\Http\Controllers\Controller;
use App\Http\Responses\DataTableJsonResponse;
use App\Models\AssetHistory;
use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Models\TechnicalSpecs;
use Facades\App\Models\Asset;
use Facades\App\Repositories\HardwareStandards;
use App\Services\Asn\Connection\ConnectionAssetsService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use App\Http\Traits\Asn\AsnAssetTrait;
use App\Models\IntegrationSyncHistory;

class ConnectionAssetsController extends Controller
{

    use AsnAssetTrait;

    /**
     * @var \App\Services\Asn\Connection\ConnectionAssetsService
     */
    protected $assetService;


    /**
     * Constructor for the controller.
     *
     * @param \App\Services\ConnectionAssetsService $assetService The asset service instance.
     */
    public function __construct(ConnectionAssetsService $assetService)
    {
        $this->assetService = $assetService;
    }


    /**
     * Display a listing of connection assets.
     *
     * @return \Illuminate\View\View The view displaying the connection assets.
     */
    public function index()
    {
        $hardwareStandards = HardwareStandards::getHardwareStandardsWithoutAccessories();
        $hardwareStandardsArray = HardwareStandards::getHardwareStandardsWithoutAccessories()->pluck('makeAndModel', 'id');
        $techSpechs = TechnicalSpecs::get()->pluck('details', 'id');
        $assettypes = AssetType::allWithoutComputerAccessories()->pluck('name', 'id');
        $lastSync = IntegrationSyncHistory::getLastSync(config('integration-sync-histories.asn-connection'))->first();
        $lastSyncDate = $lastSync?->created_at;

        return view('asn.connection.connection-assets', compact('hardwareStandards', 'assettypes', 'techSpechs', 'hardwareStandardsArray', 'lastSyncDate'));
    }


    /**
     * Retrieve filtered data for connection assets.
     *
     * @param \Illuminate\Http\Request $request The incoming HTTP request containing DataTable parameters.
     *
     * @return \App\Http\Responses\DataTableJsonResponse The JSON response containing the filtered asset data.
     */
    public function data(Request $request)
    {
        $filteredData  = $this->assetService->data('connection_in_transit');
        $assets        = $filteredData['assets'];

        $start = request('start');
        $data = [];

        if (!empty($assets)) {
            $data = $this->assetService->getAsnAssetData($assets, $start, $data);
        }

        return new DataTableJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }


    /**
     * Delete an existing connection asset.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and success message.
     */
    public function destroy()
    {
        $assetHistories = AssetHistory::where('asset_id', request('id'));
        $assetHistories->delete();

        $asset = Asset::findOrFail(request('id'));
        $asset->delete();

        $array['success'] = 'success';
        $array['status'] = 'Deleted Successfully';

        return response()->json($array);
    }


    /**
     * Handle AJAX requests for updating or deleting connection assets.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message from the update or destroy method, or a boolean value.
     */
    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }


    /**
     * Mark an asset as received.
     *
     * @return mixed The result of the `recieveAsnAsset` method, which handles the asset status update.
     */
    public function markReceived()
    {
        $oldStatus = AssetStatus::where('slug', 'connection_in_transit')->first();

        return $this->recieveAsnAsset($oldStatus);
    }


    /**
     * Export connection assets data to CSV.
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse The CSV file response.
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $data = [];
        $datas = collect();

        $filteredData = $this->assetService->getExportData('connection_in_transit')->get();
        $datas->push($this->assetService->getAsnAssetExportData($filteredData, 0, $data, 'Connection'));

        return exportToCsv($datas->toArray());
    }


    /**
     * Update an existing connection asset.
     *
     * @return \Illuminate\Http\JsonResponse|string The JSON response with validation errors or a success message.
     */
    public function updateAsset()
    {
        $error = $this->assetService->validateAsset();

        if ($error != false) {
            return response()->json($error);
        }

        $asset = Asset::findOrFail(request('id'));

        if ($asset) {
            $data = $this->getDataToUpdate();
            $asset->update($data);
        }

        return "success";
    }


    /**
     * Delete an existing connection asset.
     *
     * @param int $id The ID of the asset to be deleted.
     *
     * @return string A success message indicating the deletion status.
     */
    public function deleteAsset($id)
    {
        $asset = Asset::whereIn('asset_status_id', AssetStatus::where('slug', ['connection_in_transit'])->pluck('id'))
            ->find($id);

        if ($asset) {
            AssetHistory::where('asset_id', $asset->id)->delete();
            $asset->delete();
        }

        return "success";
    }


    /**
     * Manually synchronize connection assets.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the status and message of the synchronization.
     */
    public function manualSync()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        Artisan::call('connection:upload');

        $message = Artisan::output();

        if (substr($message, 0, 40) == "Connection assets creation completed, on") {
            return response()->json(['status' => 'success', 'message' => $message]);
        } else if (substr($message, 0, 48) == "Connection assets creation completed with errors") {
            $errors = view('asn.connection.partials.connection-sync-error-list')->render();

            return response()->json(['status' => 'error', 'message' => $message, 'errors' => $errors]);
        } else {
            return response()->json(['status' => 'error', 'message' => "Some Error Occured"]);
        }
    }
}
