<?php

namespace App\Http\Controllers\Asn\Connection;

use App\Http\Controllers\Controller;
use App\Models\AsnAccessoriesMapping;
use App\Models\Asset;
use App\Models\AssetStatus;
use Facades\App\Repositories\HardwareStandards;
use App\Services\Asn\Connection\ConnectionAccessoriesMappingService;
use App\Models\AssetHistory;

class ConnectionAccessoriesMappingController extends Controller
{

    /**
     * @var \App\Services\Asn\Connection\ConnectionAccessoriesMappingService
     */
    protected $accessoriesMappingService;


    /**
     * Constructor for the controller.
     *
     * @param \App\Services\ConnectionAccessoriesMappingService $accessoriesMappingService The accessories mapping service instance.
     */
    public function __construct(ConnectionAccessoriesMappingService $accessoriesMappingService)
    {
        $this->accessoriesMappingService = $accessoriesMappingService;
    }


    /**
     * Display a listing of connection accessories mappings.
     *
     * @return \Illuminate\View\View The view displaying the connection accessories mappings.
     */
    public function index()
    {
        $accessoriesMappings = $this->accessoriesMappingService->getAccessoriesMappingData();
        $hardwareStandards = HardwareStandards::getHardwareStandardsAccessories();
        $hardwareStandardsArray = HardwareStandards::getHardwareStandardsAccessories()->pluck('makeAndModel', 'id');

        return view('asn.connection.connection-accessories-mapping', compact('accessoriesMappings', 'hardwareStandards', 'hardwareStandardsArray'));
    }


    /**
     * Search for connection accessories mappings.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the rendered search results view.
     */
    public function search()
    {
        $accessoriesMappings = $this->accessoriesMappingService->getAccessoriesMappingSearch();

        $view['hardware'] = view('asn.connection.partials.data.connection-accessories-mapping-data', compact('accessoriesMappings'))->render();

        return response()->json($view);
    }


    /**
     * Store a new connection accessories mapping.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function store()
    {
        $accessoriesMapping = AsnAccessoriesMapping::where(['provider' => 'connection', 'part_no' => request('description')])->first();

        if (!$accessoriesMapping) {
            AsnAccessoriesMapping::Create([
                'part_no' => request('description'),
                'provider' => 'connection',
                'make_and_model_id' => request('make_and_model'),
            ]);

            return redirect('/connection-accessories-mapping')->with('message', 'Accessories mapping added successfully.');
        }

        return redirect('/connection-accessories-mapping')->withErrors(['Mapping already exist']);
    }


    /**
     * Update an existing connection accessories mapping.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message.
     */
    public function update()
    {
        AsnAccessoriesMapping::findOrFail(request('id'))
            ->update([
                'part_no' => request('description'),
                'make_and_model_id' => request('make_and_model'),
            ]);

        return response()->json(['status' => 'success', 'message' => "Accessory mapping updated successfully."]);
    }


    /**
     * Delete an existing connection accessories mapping.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message.
     */
    public function destroy()
    {
        $accessoriesMapping = AsnAccessoriesMapping::findOrFail(request('id'));
        $accessoriesMapping->delete();

        return response()->json(['status' => 'success', 'message' => "Accessory mapping deleted successfully."]);
    }


    /**
     * Handle AJAX requests for updating or deleting connection accessories mappings.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message from the update or destroy method, or a boolean value.
     */
    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }


    /**
     * Resync connection accessories mappings.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success message.
     */
    public function resync()
    {
        $allAccessoriesMappings = AsnAccessoriesMapping::where('provider', 'connection')->whereNotNull('part_no')->get();

        foreach ($allAccessoriesMappings as $accessoriesMapping) {
            $acessories = Asset::whereIn('asset_status_id', AssetStatus::whereIn('slug', ['connection_in_transit'])->pluck('id'))->where('part_no', $accessoriesMapping->part_no)
                ->get();

            foreach ($acessories as $accessory) {
                AssetHistory::where('asset_id', $accessory->id)->delete();
                $accessory->delete();
            }
        }

        return redirect('/connection-accessories-mapping')->with('message', "Resynced Successfully");
    }


    /**
     * Handle the bulk upload of connection accessories mappings.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function bulkUpload()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $result = $this->accessoriesMappingService->bulkUpload();

        if (isset($result['errors'])) {
            return redirect('/connection-accessories-mapping')->with(['message' => $result['message'] ?? '', 'error' => $result['errors'] ?? '']);
        }

        return redirect('/connection-accessories-mapping')->with(['message' => $result['message'] ?? '']);
    }
}
