<?php

namespace App\Http\Controllers\Asn\Cdw;

use App\Http\Requests\Asn\Cdw\StoreCdwUnwantedParts;
use App\Services\Asn\Cdw\CdwUnwantedPartsService;
use App\Http\Controllers\Controller;
use App\Models\AsnUnwantedPart;
use App\Models\AssetHistory;
use App\Models\AssetStatus;
use App\Models\Asset;

class CdwUnwantedPartsController extends Controller
{

    /**
     * @var \App\Services\CdwUnwantedPartsService
     */
    protected $unwantedPartsService;


    /**
     * Constructor for the controller.
     *
     * @param \App\Services\CdwUnwantedPartsService $unwantedPartsService The unwanted parts service instance.
     */
    public function __construct(CdwUnwantedPartsService $unwantedPartsService)
    {
        $this->unwantedPartsService = $unwantedPartsService;
    }


    /**
     * Display a listing of unwanted parts.
     *
     * @return \Illuminate\View\View The view displaying the unwanted parts.
     */
    public function index()
    {
        $unwantedParts = $this->unwantedPartsService->getUnwantedPartsData();

        return view('asn.cdw.cdw-unwanted-parts', compact('unwantedParts'));
    }


    /**
     * Search for unwanted parts.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the rendered search results view.
     */
    public function search()
    {
        $unwantedParts = $this->unwantedPartsService->getUnwantedPartsSearch();

        $view['hardware'] = view('asn.cdw.partials.data.cdw-unwanted-parts-data', compact('unwantedParts'))->render();

        return response()->json($view);
    }


    /**
     * Store a new CDW unwanted part mapping.
     *
     * @param \App\Http\Requests\StoreCdwUnwantedParts $request The incoming HTTP request validated using the StoreCdwUnwantedParts request class.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function store(StoreCdwUnwantedParts $request)
    {
        $mappingData = AsnUnwantedPart::where(['provider' => 'cdw', 'part_no' => $request->description])->first();

        if (!$mappingData) {
            AsnUnwantedPart::Create([
                'part_no' => request('description'),
                'provider' => 'cdw',
            ]);

            return redirect('/cdw-unwanted-parts')->with('message', 'Mapping added successfully.');
        }

        return redirect('/cdw-unwanted-parts')->withErrors(['Mapping already exist']);
    }


    /**
     * Update an existing CDW unwanted part mapping.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message.
     */
    public function update()
    {
        AsnUnwantedPart::findOrFail(request('id'))
            ->update([
                'part_no' => request('description')
            ]);

        return response()->json(['status' => 'success', 'message' => "Unwanted parts mapping updated successfully."]);
    }


    /**
     * Delete an existing CDW unwanted part mapping.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message.
     */
    public function destroy()
    {
        $mappingData = AsnUnwantedPart::findOrFail(request('id'));
        $mappingData->delete();

        return response()->json(['status' => 'success', 'message' => "Unwanted parts mapping deleted successfully."]);
    }


    /**
     * Handle AJAX requests for updating or deleting CDW unwanted parts mappings.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message from the update or destroy method, or a boolean value.
     */
    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }


    /**
     * Handle the bulk upload of CDW unwanted parts mappings.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function bulkUpload()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $result = $this->unwantedPartsService->bulkUpload();

        if (isset($result['errors'])) {
            return redirect('/cdw-unwanted-parts')->with(['message' => $result['message'] ?? '', 'error' => $result['errors'] ?? '']);
        }

        return redirect('/cdw-unwanted-parts')->with(['message' => $result['message'] ?? '']);
    }


    /**
     * Resync CDW unwanted parts mappings.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success message.
     */
    public function resync()
    {
        $allMappings = AsnUnwantedPart::where('provider', 'cdw')->whereNotNull('part_no')->get();
        $status      = AssetStatus::whereIn('slug', ['cdw_in_transit'])->first();

        foreach ($allMappings as $mapping) {
            $assets = Asset::select('id')
                ->where('asset_status_id', $status->id)
                ->where('part_no', $mapping->part_no)
                ->get();

            foreach ($assets as $asset) {
                AssetHistory::where('asset_id', $asset->id)->delete();
                $asset->delete();
            }
        }

        return redirect('/cdw-unwanted-parts')->with('message', "Resynced Successfully");
    }
}
