<?php

namespace App\Http\Controllers\Asn\Cdw;

use App\Services\Asn\Cdw\CdwHardwareMappingService;
use Facades\App\Repositories\HardwareStandards;
use App\Http\Requests\Asn\Cdw\StoreCdwHardwareMapping;
use App\Http\Controllers\Controller;
use Facades\App\Models\MakeAndModel;
use App\Models\AsnHardwareMapping;
use App\Models\TechnicalSpecs;
use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Models\Asset;

class CdwHardwareMappingController extends Controller
{

    /**
     * @var CdwHardwareMappingService
     */
    protected $hardwareMappingService;



    /**
     * HardwareMappingController constructor.
     *
     * @param CdwHardwareMappingService $hardwareMappingService The hardware mapping service instance.
     *
     */
    public function __construct(CdwHardwareMappingService $hardwareMappingService)
    {
        $this->hardwareMappingService = $hardwareMappingService;
    }


    /**
     * Display a listing of CDW hardware mappings.
     *
     * @return \Illuminate\View\View The view displaying the CDW hardware mappings.
     */
    public function index()
    {
        $cdwMappings            = $this->hardwareMappingService->getCdwMappingData();
        $hardwareStandards      = HardwareStandards::getHardwareStandardsWithoutAccessories();
        $hardwareStandardsArray = HardwareStandards::getHardwareStandardsWithoutAccessories()->pluck('makeAndModel', 'id');
        $assettypes             = AssetType::pluck('name', 'id');
        $techSpechs             = TechnicalSpecs::get()->pluck('details', 'id');

        return view('asn.cdw.cdw-hardware-mapping', compact('cdwMappings', 'assettypes', 'hardwareStandards', 'hardwareStandardsArray', 'techSpechs'));
    }


    /**
     * Search for CDW hardware mappings.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the rendered search results view.
     */
    public function search()
    {
        $cdwMappings      = $this->hardwareMappingService->filter();
        $view['hardware'] = view('asn.cdw.partials.data.cdw-hardware-mapping-data', compact('cdwMappings'))->render();

        return response()->json($view);
    }


    /**
     * Store a new CDW hardware mapping.
     *
     * @param \App\Http\Requests\StoreCdwHardwareMapping $request The incoming HTTP request validated using the StoreCdwHardwareMapping request class.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function store(StoreCdwHardwareMapping $request)
    {
        $venderMappingExist = AsnHardwareMapping::where(['provider' => 'cdw', 'part_no' => $request->description])->first();

        if ($venderMappingExist) {
            return redirect('/cdw-hardware-mapping')->withErrors(['Mapping for this description already exist']);
        }

        $this->hardwareMappingService->store();

        return redirect('/cdw-hardware-mapping')->with('message', "Hardware Mapped Successfully");
    }


    /**
     * Handle AJAX requests for updating or deleting CDW hardware mappings.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message from the update or destroy method, or a boolean value.
     */
    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }


    /**
     * Update an existing CDW hardware mapping.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message.
     */
    public function update()
    {
        $techSpec  = TechnicalSpecs::where('id', request('technical_spec'))->first();
        $makeModel = MakeAndModel::where('id', request('make_and_model'))->first();

        AsnHardwareMapping::findOrFail(request('id'))
            ->update([
                'make_and_model_id' => $makeModel ? $makeModel->id : null,
                'technical_spec_id' => $techSpec ? $techSpec->id : null,
                'part_no'       => request('description'),
            ]);

        return response()->json(['status' => 'success', 'message' => "Hardware mapping updated successfully."]);
    }


    /**
     * Delete an existing CDW hardware mapping.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response with the status and message.
     */
    public function destroy()
    {
        $asnHardwareMapping = AsnHardwareMapping::findOrFail(request('id'));
        $assetStatus = AssetStatus::whereIn('slug', ['cdw_in_transit'])->pluck('id');

        $this->hardwareMappingService->resyncHardwarePartOnDelete($assetStatus, $asnHardwareMapping);
        $asnHardwareMapping->delete();

        return response()->json(['status' => "success", 'message' => "Hardware mapping deleted successfully."]);
    }


    /**
     * Bulk upload mappingHandle the bulk upload of CDW hardware mappings.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success or error message.
     */
    public function bulkUpload()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $result = $this->hardwareMappingService->bulkUpload();

        return redirect('/cdw-hardware-mapping')->with(['message' => $result['message'] ?? '', 'error' => $result['errors'] ?? '']);
    }


    /**
     * Resync CDW hardware mappings.
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response with a success message.
     */
    public function resync()
    {
        $allCdwHardwareMappings = AsnHardwareMapping::where('provider', 'cdw')->whereNotNull('part_no')->get();

        $assetStatusId = AssetStatus::whereIn('slug', ['cdw_in_transit'])->pluck('id');

        foreach ($allCdwHardwareMappings as $cdwHardwareMapping) {
            $makeAndModel  = MakeAndModel::findOrFail($cdwHardwareMapping->make_and_model_id);

            Asset::whereIn('asset_status_id', $assetStatusId)
                ->where('part_no', $cdwHardwareMapping->part_no)
                ->update([
                    'make_and_model_id' => $cdwHardwareMapping->make_and_model_id,
                    'technical_spec_id' => $cdwHardwareMapping->technical_spec_id,
                    'asset_type_id'     => $makeAndModel->asset_type_id
                ]);
        }

        return redirect('/cdw-hardware-mapping')->with('message', "Resynced successfully.");
    }
}
