<?php

namespace App\Http\Controllers\Asn\Cdw;

use App\Http\Responses\DataTableJsonResponse;
use App\Services\Asn\Cdw\CdwAssetsService;
use App\Http\Controllers\Controller;
use App\Models\TechnicalSpecs;
use Illuminate\Http\Request;
use App\Models\AssetHistory;
use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Models\Asset;
use Facades\App\Repositories\HardwareStandards;
use App\Http\Traits\Asn\AsnAssetTrait;
use App\Models\IntegrationSyncHistory;
use Illuminate\Support\Facades\Artisan;

class CdwAssetsController extends Controller
{

    use AsnAssetTrait;


    /**
     * Constructor for ReceiveCdwAssetsController
     *
     * @param CdwAssetsService $assetService
     */
    public function __construct(protected CdwAssetsService $assetService) {}


    /**
     * Display a listing of CDW assets.
     *
     * @return \Illuminate\View\View The view displaying the CDW assets.
     */
    public function index()
    {
        $hardwareStandards  = HardwareStandards::getHardwareStandardsWithoutAccessories();
        $hardwareStandardsArray = HardwareStandards::getHardwareStandardsWithoutAccessories()->pluck('makeAndModel', 'id');
        $assettypes         = AssetType::allWithoutComputerAccessories()->pluck('name', 'id');
        $techSpechs         = TechnicalSpecs::get()->pluck('details', 'id');
        $lastSync = IntegrationSyncHistory::getLastSync(config('integration-sync-histories.asn-cdw'))->first();
        $lastSyncDate = $lastSync?->created_at;

        return view('asn.cdw.cdw-assets', compact('assettypes', 'hardwareStandards', 'hardwareStandardsArray', 'techSpechs', 'lastSyncDate'));
    }


    /**
     * Retrieve filtered data for CDW assets.
     *
     * @param \Illuminate\Http\Request $request The incoming HTTP request containing DataTable parameters.
     *
     * @return \App\Http\Responses\DataTableJsonResponse The JSON response containing the filtered asset data.
     */
    public function data(Request $request)
    {
        $filteredData  = $this->assetService->data('cdw_in_transit');
        $assets        = $filteredData['assets'];

        $start = request('start');
        $data = [];

        if (!empty($assets)) {
            $data = $this->assetService->getAsnAssetData($assets, $start, $data);
        }

        return new DataTableJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }


    /**
     * Export CDW assets data to CSV.
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse The CSV file response.
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $data = [];
        $datas = collect();
        $filteredData = $this->assetService->getExportData('cdw_in_transit')->get();
        $datas->push($this->assetService->getAsnAssetExportData($filteredData, 0, $data, 'CDW'));

        return exportToCsv($datas->toArray());
    }


    /**
     * Update an existing asset.
     *
     * @return \Illuminate\Http\JsonResponse|string The JSON response with validation errors or a success message.
     */
    public function updateAsset()
    {
        $error = $this->assetService->validateAsset();

        if ($error != false) {
            return response()->json($error);
        }

        $asset = Asset::findOrFail(request('id'));

        if ($asset) {
            $data = $this->getDataToUpdate();
            $asset->update($data);
        }

        return "success";
    }


    /**
     * Delete an existing asset.
     *
     * @param int $id The ID of the asset to be deleted.
     *
     * @return string A success message indicating the deletion status.
     */
    public function deleteAsset($id)
    {
        $assetStatusId = AssetStatus::where('slug', 'cdw_in_transit')->pluck('id');
        $asset         = Asset::where('asset_status_id', $assetStatusId)->find($id);

        if (!$asset) {
            return "success";
        }

        $asset->delete();

        AssetHistory::where('asset_id', $id)->delete();

        return "success";
    }

    /**
     * Mark an asset as received.
     *
     * @return mixed The result of the `recieveAsnAsset` method, which handles the asset status update.
     */
    public function markReceived()
    {
        $oldStatus = AssetStatus::where('slug', 'cdw_in_transit')->first();

        return $this->recieveAsnAsset($oldStatus);
    }

    /**
     * Manually synchronize CDW assets.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the status and message of the synchronization.
     */
    public function manualSync()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        Artisan::call('cdw:upload');
        $message = Artisan::output();

        if (substr($message, 0, 33) == "CDW Assets creation completed, on") {
            return response()->json(['status' => 'success', 'message' => $message]);
        } else if (substr($message, 0, 41) == "CDW assets creation completed with errors") {
            $errors = view('asn.cdw.partials.cdw-sync-error-list')->render();

            return response()->json(['status' => 'error', 'message' => $message, 'errors' => $errors]);
        } else {
            return response()->json(['status' => 'error', 'message' => $message]);
        }
    }
}
