<?php

namespace App\Http\Controllers\Api\Scim;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\User;
use App\Services\Api\Scim\ScimUserService;
use Illuminate\Support\Facades\Log;

class UserController extends Controller
{


    /**
     * Constructor for the controller.
     *
     * @param \App\Services\ScimUserService $userService The SCIM user service instance.
     */
    public function __construct(protected ScimUserService $userService)
    {
    }


    /**
     * Retrieve SCIM users with pagination.
     *
     * @param \Illuminate\Http\Request $request The incoming HTTP request.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the paginated list of users.
     */
    public function index(Request $request)
    {
        $startIndex = $request->query('startIndex', 1);
        $count = $request->query('count', 2);

        $users = [];

        $response = [
            'schemas' => ['urn:ietf:params:scim:api:messages:2.0:ListResponse'],
            'totalResults' => 0,
            'startIndex' => $startIndex,
            'itemsPerPage' => $count,
            'Resources' => $users
        ];

        return response()->json($response);
    }


    /**
     * Create a new user.
     *
     * @param \Illuminate\Http\Request $request The incoming HTTP request containing the SCIM data.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the created user data..
     */
    public function create(Request $request)
    {
        $scimData = $request->all();
        Log::channel('daily')->info('SCIM user data: ' . json_encode($scimData));

        return response()->json($this->userService->create($scimData));
    }


    /**
     * Update an existing user.
     *
     * @param \Illuminate\Http\Request $request The incoming HTTP request containing the SCIM data.
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the updated user data.
     */
    public function update(Request $request)
    {
        $scimData = $request->all();
        Log::channel('daily')->info('SCIM user data: ' . json_encode($scimData));

        return response()->json($this->userService->update($scimData));
    }


    /**
     * Retrieve a user by ID.
     *
     * @param int $id The ID of the user to be retrieved.
     *
     * @return \Illuminate\Http\JsonResponse|array The JSON response containing the user data or an error response.
     */
    public function getUser($id)
    {
        $user = User::find($id);

        if (!$user) {
            return [
                'schemas' => ['urn:ietf:params:scim:api:messages:2.0:Error'],
                'status' => 404,
                'detail' => 'User not found.'
            ];
        }

        return response()->json([
            'id' => $id,
            'userName' => $user->email,
            'active' => true,
            'schemas' => ['urn:ietf:params:scim:schemas:core:2.0:User']
        ]);
    }
}
