<?php

namespace App\Http\Controllers\Api;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\Api\GetLocationRequest;
use App\Services\Api\LocationApiService;

/**
 * Controller class for Location API
 */
class LocationApiController extends Controller
{
    /**
     * @var LocationApiService
     */
    protected $locationApiService;

    /**
     * Constructor to initialize the LocationApiController.
     *
     * @param LocationApiService $locationApiService
     */
    public function __construct(LocationApiService $locationApiService)
    {
        $this->locationApiService = $locationApiService;
    }

    /**
     * Get all locations.
     *
     * @param GetLocationRequest $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getLocations(GetLocationRequest $request)
    {
        if ($this->locationApiService->checkDataLimit($request->limit)) {
            return $this->locationApiService->limitExceededResponse();
        }

        $locations = $this->locationApiService->getLocations($request);

        if ($locations->count() == 0) {
            return $this->locationApiService->noResultResponse('No Locations Found');
        }

        return $this->locationApiService->successResponse($locations);
    }


    /**
     * Search locations using simple search.
     *
     * @param Request $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function searchLocation(Request $request)
    {
        $request->validate([
            'search' => ['required', 'regex:/^[a-z0-9,@. _-]+$/i']
        ]);

        if ($this->locationApiService->checkDataLimit($request->limit)) {
            return $this->locationApiService->limitExceededResponse();
        }

        $locations = $this->locationApiService->searchLocations($request);

        if ($locations->count() == 0) {
            return $this->locationApiService->noResultResponse('No Locations Found');
        }

        return $this->locationApiService->successResponse($locations);
    }
}
