<?php

namespace App\Http\Controllers\Api;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\Api\GetHardwareRequest;
use App\Services\Api\HardwareStandardApiService;

/**
 * Controller class for Hardware Standard API
 */
class HardwareStandardApiController extends Controller
{

    /**
     * @var HardwareStandardApiService
     */
    protected $hardwareApiService;

    /**
     * Create a new HardwareStandardApiController instance.
     *
     * @param HardwareStandardApiService $hardwareApiService
     */
    public function __construct(HardwareStandardApiService $hardwareApiService)
    {
        $this->hardwareApiService = $hardwareApiService;
    }

    /**
     * Get all hardwares.
     *
     * @param GetHardwareRequest $request The HTTP request.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getHardwares(GetHardwareRequest $request)
    {
        if ($this->hardwareApiService->checkDataLimit($request->limit)) {
            return $this->hardwareApiService->limitExceededResponse();
        }

        $hardwares = $this->hardwareApiService->getHardwares($request);

        if ($hardwares->count() == 0) {
            return $this->hardwareApiService->noResultResponse('Hardware Standards not found');
        }

        return $this->hardwareApiService->successResponse($hardwares);
    }


    /**
     * Search hardwares using simple search.
     *
     * @param Request $request The HTTP request.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function searchHardwares(Request $request)
    {
        $request->validate([
            'search' => ['required', 'regex:/^[a-z0-9,@. _-]+$/i']
        ]);

        if ($this->hardwareApiService->checkDataLimit($request->limit)) {
            return $this->hardwareApiService->limitExceededResponse();
        }

        $hardwares = $this->hardwareApiService->searchHardwareStandards($request);

        if ($hardwares->count() == 0) {
            return $this->hardwareApiService->noResultResponse('Hardware Standards not found');
        }

        return $this->hardwareApiService->successResponse($hardwares);
    }
}
