<?php

namespace App\Http\Controllers\Api;

use JWTAuth;
use Tymon\JWTAuth\Exceptions\JWTException;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\ApiClient;
use Laravel\Passport\Client;
use GuzzleHttp\Exception\ClientException;

/**
 * Controller class for authentication.
 */
class AuthenticateController extends Controller
{
    /**
     * Authenticate a user and generate a JWT token.
     *
     * @param Request $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function authenticate(Request $request)
    {
        // grab credentials from the request
        $credentials = $request->only('email', 'password');

        try {
            // attempt to verify the credentials and create a token for the user
            if (! $token = JWTAuth::attempt($credentials)) {
                return response()->json(['error' => 'invalid_credentials'], 401);
            }
        } catch (JWTException $e) {
            // something went wrong whilst attempting to encode the token
            return response()->json(['error' => 'could_not_create_token'], 500);
        }

        // all good so return the token
        return response()->json(compact('token'));
    }

    /**
     * Refresh an access token.
     *
     * @param Request $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function refreshToken(Request $request)
    {
        $validator = Validator::make(request()->all(), [
            'client_id' => 'required',
            'client_secret' => 'required',
        ]);

        if ($validator->fails()) {
            $data = [
                'error'       	=> true,
                'message'       => 'Error',
                'status_code'   => 404,
                'errors' 		=> $validator->errors()
            ];

            return response()->json($data);
        }

        $client = Client::where(function ($query) use ($request) {
            $query->where('id', $request->client_id)
              ->orWhere('uuid', $request->client_id);
        })->where('secret', $request->client_secret)->first();


        if ($client) {
            $client->tokens()->update(['revoked' => true]);
            $return['token'] = $this->generateToken($client);

            if ($return['token'] == "auth_failed") {
                $data = [
                    'success'       => true,
                    'message'       => 'Client authentication failed',
                    'status_code'   => 401,
                    'data' => ''
                ];

                return response()->json($data);
            }

            $validity = explode(' ', $client->validity);

            $client->tokens()->whereRevoked(false)->update(['expires_at' => now()->add($validity[0], $validity[1])]);

            $data = [
                'success'       => true,
                'message'       => 'Created Token',
                'status_code'   => 200,
                'data' => $return
            ];

            return response()->json($data);
        }

        $data = [
            'success'     => false,
            'message'     => 'Unauthorized access',
            'status_code' => 401,
        ];

        return response()->json($data);
    }

    /**
     * Generate a token for the client.
     *
     * @param Client $client
     *
     * @return string
     */
    public function generateToken($client)
    {
        $scopes = "";

        $apiClient = ApiClient::find($client->id);

        if ($apiClient->oAuthscopes) {
            $scopes = implode(' ', $apiClient->oAuthscopes->pluck('slug')->toArray());
        }

        try {

            $response = (new \GuzzleHttp\Client)->post(url('oauth/token'), [
                'form_params' => [
                    'grant_type' => 'client_credentials',
                    'client_id' => $client->id, // Client ID
                    'client_secret' => $client->secret, // Client secret
                    'scope' => $scopes
                ]
            ]);

            return json_decode((string) $response->getBody(), true)['access_token'];
        } catch (ClientException $e) {
            return "auth_failed";
        }
    }
}
