<?php

namespace App\Http\Controllers\Api;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\Api\GetAssetRequest;
use App\Services\Api\SearchAssetService;

/**
 * Controller class for Asset Search API.
 */
class AssetSearchApiController extends Controller
{
    /**
     * @var SearchAssetService
     */
    protected $searchAssetService;

    /**
     * Create a new controller instance.
     *
     * @param SearchAssetService $searchAssetService
     */
    public function __construct(SearchAssetService $searchAssetService)
    {
        $this->searchAssetService = $searchAssetService;
    }

    /**
     * Search assets using simple search.
     *
     * @param Request $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function searchAsset(Request $request)
    {
        $request->validate([
            'search' => ['required', 'regex:/^[a-z0-9,@. _-]+$/i'],
            'limit'  => ['nullable', 'numeric'],
            'page'   => ['nullable', 'numeric'],
        ]);

        if ($this->searchAssetService->checkDataLimit($request->limit)) {
            return $this->searchAssetService->limitExceededResponse();
        }

        $assets = $this->searchAssetService->searchAssets($request);

        if ($assets->count() == 0) {
            return $this->searchAssetService->noResultResponse('No Assets Found');
        }

        return $this->searchAssetService->successResponse($assets);
    }

    /**
     * Get all assets.
     *
     * @param GetAssetRequest $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getAllAssets(GetAssetRequest $request)
    {

        if ($this->searchAssetService->checkDataLimit($request->limit)) {
            return $this->searchAssetService->limitExceededResponse();
        }

        $assets = $this->searchAssetService->getAssets($request);

        if ($assets->count() == 0) {
            return $this->searchAssetService->noResultResponse('No Assets Found');
        }

        return $this->searchAssetService->successResponse($assets);
    }

    /**
     * Get terminated user assets.
     *
     * @param GetAssetRequest $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getTerminatedUserAssets(GetAssetRequest $request)
    {

        if ($this->searchAssetService->checkDataLimit($request->limit)) {
            return $this->searchAssetService->limitExceededResponse();
        }

        $assets = $this->searchAssetService->getTerminatedUserAssets($request);

        if ($assets->count() == 0) {
            return $this->searchAssetService->noResultResponse('No Assets Found');
        }

        return $this->searchAssetService->successResponse($assets);
    }
}
