<?php

namespace App\Http\Controllers;

use App\Events\BulkUpdates;
use Illuminate\Http\Request;
use App\User;
use App\Models\UserType;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class AdminUserController extends Controller
{

    public function index()
    {
        $adminuser = UserType::admin()->pluck('name', 'id');
        $users = User::with('userType')->admin()->get();

        return view('settings.admin-users', compact('users', 'adminuser'));
    }

    public function store(Request $request)
    {

        $this->validate(request(), [
            'first_name'   => 'required',
            'last_name'    => 'required',
            'email'        => 'required|unique:users',
            'user_type_id' => 'required',
        ]);

        $userExists = User::where('email', $request['email'])->first();

        if (!$userExists) {
            $request = $request->only(['first_name', 'last_name', 'email', 'user_type_id']);

            $user = User::create($request);
            $user->assignRole($user->userType->name);
            $userName = $user->first_name . ' ' . $user->last_name;
            $description = __('history.AdminCreated', [
                'userId'    => $user->id,
                'userName'  => $userName,
            ]);

            $systemHistory = [
                'user_id'     => Auth::id(),
                'new_user_id' => $user->id,
                'action'      => 'admin_created',
                'description' => $description,
                'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            ];

            event(new BulkUpdates($systemHistory));

            return redirect(route('adminusers.index'))->with('message', __('message.user_created'));
        } else {
            return redirect('/adminusers')->withErrors(['Email already exists']);
        }
    }

    public function update()
    {
        $user = User::findOrFail(request('id'));

        $oldFirstName = $user->first_name;
        $oldLastName  = $user->last_name;
        $oldEmail     = $user->email;
        $userType     = UserType::where('id', $user->user_type_id)->first();
        $oldUserType  = $userType->name;

        $user->update([
            'first_name'   => request('first_name'),
            'last_name'    => request('last_name'),
            'email'        => request('email'),
            'user_type_id' => request('usertype'),
        ]);

        $user->syncRoles($user->userType->name);

        $newFirstName = request('first_name');
        $newLastName  = request('last_name');
        $newEmail     = request('email');
        $userType     = UserType::where('id', request('usertype'))->first();
        $newUserType  = $userType->name;

        $description = '';

        if ($oldFirstName != $newFirstName) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'First Name',
                'oldValue'    => $oldFirstName,
                'newValue'  => $newFirstName,
            ]);
        }

        if ($oldLastName != $newLastName) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Last Name',
                'oldValue'    => $oldLastName,
                'newValue'  => $newLastName,
            ]);
        }

        if ($oldEmail != $newEmail) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'Email',
                'oldValue'    => $oldEmail,
                'newValue'  => $newEmail,
            ]);
        }

        if ($oldUserType != $newUserType) {
            $description .= __('history.FieldsUpdated', [
                'fieldName' => 'User Type',
                'oldValue'    => $oldUserType,
                'newValue'  => $newUserType,
            ]);
        }

        $description = __('history.AdminUpdated', [
            'description' => $description
        ]);

        $systemHistory = [
            'user_id'     => Auth::id(),
            'new_user_id' => $user->id,
            'action'      => 'admin_updated',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        event(new BulkUpdates($systemHistory));

        return response()->json('success');
    }

    public function destroy()
    {
        $user = User::findOrFail(request('id'));
        $userName = $user->first_name . ' ' . $user->last_name;
        $user->delete();

        $description = __('history.AdminDeleted', [
            'userName' => $userName
        ]);

        $assetHistory = [
            'user_id' => Auth::id(),
            'action' => 'admin_deleted',
            'description' => $description,
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        event(new BulkUpdates($assetHistory));

        return response()->json(__('message.deleted'));
    }

    public function ajax()
    {
        if (request('action') == 'edit') {
            return $this->update();
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }
}
