<?php

namespace App\Console\Commands;

use App\Models\CronJob;
use App\Notifications\Integrations\ZoomSyncNotification;
use App\Services\SoftwareLicense\ImportedAssets\Zoom\ZoomService;
use Carbon\Carbon;
use Exception;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;

class ZoomSync extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'Zoom:sync';
    protected $name;

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sync the zoom info and member list';

    private $service;

    /**
     * Create a new command instance
     *
     * @param ZoomService $service
     * 
     * @return void
     */
    public function __construct(ZoomService $service)
    {
        parent::__construct();
        $this->name = 'Zoom';
        $this->service = $service;
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        try {
            $credentials = $this->service->getCredentials();

            if (!$credentials) {
                $this->info(json_encode([
                    'status'  => 'error',
                    'message' => $this->name . ' credentials are not found.'
                ]));

                return;
            }

            $assetDetails = $this->service->getLicenseData();

            if (!$assetDetails) {
                $this->info(json_encode([
                    'status'  => 'error',
                    'message' => $this->name . ' asset not created yet.'
                ]));

                return;
            }

            if (!$this->service->checkConnection($credentials)) {
                $connectionStatus = $this->service->setConnection();

                if ($connectionStatus !== true) {
                    $this->info(json_encode([
                        'status'  => 'error',
                        'message' => $this->name . ' credentials are invalid.'
                    ]));

                    return;
                }
            }


            $status = 'error';
            $message = 'An error occured with ' . $this->name . '  sync';
            $syncActionResponseStatus = $this->syncActions($credentials, $assetDetails);

            if ($syncActionResponseStatus['status']) {
                $status = 'success';
                $message = $this->name . ' sync completed successfully';
                $this->sendNotification(true);
            }


            $this->info(json_encode([
                'status'  => $status,
                'message' => $message
            ]));


            return;
        } catch (Exception $e) {
            Log::channel('daily')->info($this->name . " sync error: " . $e->getMessage());
            $this->sendNotification(false);
            $this->info(json_encode([
                'status'  => 'success',
                'message' => $this->name . ' sync failed.'
            ]));

            return;
        }
    }

    /**
     * Software aseet api sync actions
     * 
     * @param object $credentials
     * @param object $assetDetails
     * 
     * @return array 
     */
    private function syncActions($credentials, $assetDetails)
    {
        $status = true;
        $message = $this->name . ' sync completed successfully';

        $updateLicencePurchasedCountStatus = $this->service->updateLicencePurchasedCount($credentials, $assetDetails);

        if (!$updateLicencePurchasedCountStatus) {
            $status = false;
            $message = 'An error occured with update on licence purchased Count';
        }

        $createSoftwareAssetMembersStatus = $this->service->createMembers($credentials);

        if (!$createSoftwareAssetMembersStatus) {
            $status = false;
            $message = 'An error occured with adding ' . $this->name . ' members';
        }

        $softwareAssetMemberListSyncStatus = $this->service->softwareAssetMemberListSync($assetDetails);

        if (!$softwareAssetMemberListSyncStatus) {
            $status = false;
            $message = 'An error occured with ' . $this->name . ' member list sync';
        }

        $updateSoftwareOwnerStatus = $this->service->updateSoftwareAssetOwner($assetDetails);

        if (!$updateSoftwareOwnerStatus) {
            $status = false;
            $message = 'An error occured with update of ' . $this->name . ' owner';
        }

        $updateLicensesUsedCountStatus = $this->service->updateLicensesUsedCount($assetDetails);

        if (!$updateLicensesUsedCountStatus) {
            $status = false;
            $message = 'An error occured with update of ' . $this->name . ' used license count';
        }

        $toDate = Carbon::now()->format('Y-m-d');
        $fromDate = $this->getLoggedData();
        $zoomMeetingSyncStatus = $this->service->zoomMeetingsListSync($fromDate, $toDate);

        if (!$zoomMeetingSyncStatus) {
            $status = false;
            $message = 'An error occured with ' . $this->name . ' meetings list sync';
        }

        $this->updateCronJob($fromDate, $toDate);

        return ['status' => $status, 'message' => $message];
    }

    /**
     * Get the last updated date
     *
     * @return date
     */
    public function getLoggedData()
    {
        $dateFrom = Carbon::now()->subYear();

        $loggedData = CronJob::where('action', 'zoom_meetings')->orderBy('created_at', 'desc')->first();
        if ($loggedData) {
            $dateFrom = Carbon::parse($loggedData->date_to);
        }

        return $dateFrom->format('Y-m-d');
    }

    /**
     * Updates a cron job.
     *
     * @param mixed $dateFrom  The start date for the cron job.
     * @param mixed $dateTo    The end date for the cron job.
     *
     * @throws Some_Exception_Class Description of exception
     *
     * @return CronJob The updated cron job.
     */
    public function updateCronJob($dateFrom, $dateTo)
    {
        $cronData = [
            'action' => 'zoom_meetings',
            'started_at' => Carbon::now()->format('Y-m-d H:i:s'),
            'ended_at' => Carbon::now(),
            'date_from' => $dateFrom,
            'date_to' => $dateTo,
        ];
        return CronJob::create($cronData);
    }

    /**
     *  It sends a notification to a Slack channel using the Slack webhook URL
     * 
     * @param bool status true or false
     */
    public function sendNotification(bool $status)
    {
        Notification::route('slack', config('teqtivity-notification-mails.slack'))->notify(new ZoomSyncNotification($status));
    }
}
