<?php

namespace App\Console\Commands;

use App\Notifications\Slack;
use App\Services\Zoom\ZoomHardwareCredentialsService;
use App\Services\Zoom\ZoomHardwareSyncService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Notification;

class ZoomHardwareDataSync extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'zoomHardwareDataSync:run';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sync Zoom room details and associated devices from the Zoom API to the database.';

    /**
     * Create a new command instance.
     *
     * @param ZoomHardwareSyncService $syncService
     * @param ZoomHardwareCredentialsService $credentialsService
     *
     * @return void
     */
    public function __construct(
        protected ZoomHardwareSyncService $syncService,
        protected ZoomHardwareCredentialsService $credentialsService
    ) {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        if ($this->credentialsService->checkApiConnection() === false) {
            $response = 'Zoom Hardware API connection failed!';
            $this->sendNotification($response);

            return $this->error($response);
        }

        $response = $this->syncService->syncZoomRooms();

        // Reinstall location discrepancy assets
        $this->syncService->reInstallLocationDiscrepancyAssets();

        if ($response === true) {
            $this->sendNotification("Zoom Hardware API sync completed successfully.");

            return $this->info("Zoom Hardware API sync completed successfully.");
        }

        $errorResponse = 'Zoom Hardware API sync failed!';
        $this->sendNotification($errorResponse);

        return $this->error($errorResponse);
    }

    /**
     * Send Slack Notification
     *
     * @param string $message
     */
    public function sendNotification(string $message)
    {
        if (config('error-notification-mails.slack')) {
            Notification::route('slack', config('error-notification-mails.slack'))->notify(new Slack($message));
        }
    }
}
