<?php

namespace App\Console\Commands;

use App\Events\IntegrationSyncHistory;
use App\Services\Integrations\WorkspaceOneApi;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use App\Models\ApiCredential;
use App\Notifications\Slack;
use Illuminate\Support\Facades\Schema;
use Artisan;

class WorkspaceOneUpload extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'WorkspaceOneUpload:upload';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Upload the WorkspaceOne API data to the Database';


    /**
     * Class constructor.
     *
     * @param \App\Services\WorkspaceOneApi $service The WorkspaceOneApi service instance.
     */
    public function __construct(protected WorkspaceOneApi $service)
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $workspaceOneApis = ApiCredential::where('slug', 'like', 'airwatch%')->get();

        foreach ($workspaceOneApis as $api) {
            $this->info('Start time ' . date('Y-m-d H:i:s'));

            if (!$this->service->apiConnect($api->slug)) {
                $error = "FAILED! $api->name connection failed.";
                $this->error($error);
                $this->sendNotification($error);
                continue;
            }

            $this->info("Working " . $api->slug);

            Log::channel('single')->info("$api->api_name file upload started.");

            $this->service->upload($api->slug);
            $this->autoResassignDiscrepacyDevices();
            $message = "$api->api_name file upload finished.";

            Log::channel('daily')->info($message);

            $this->sendNotification($message);
            $this->info('End time ' . date('Y-m-d H:i:s'));

            // Event to add sync history(sync_name, sync_status, notes).
            event(new IntegrationSyncHistory('workspace-one'));
        }
    }


    /**
     * Re assign the user for user mismatch devices.
     * @return [type]
     */
    private function autoResassignDiscrepacyDevices()
    {
        return $this->service->reassignAssets();
    }


    /**
     * Send Mail Notification
     *
     * @param mixed $message
     */
    public function sendNotification($message)
    {
        Notification::route('slack', config('teqtivity-notification-mails.slack'))->notify(new Slack($message));

        return true;
    }
}
