<?php

namespace App\Console\Commands;

use App\Models\AssetStatus;
use App\Models\SlackNotificationCredential;
use App\Notifications\TerminatedUsersStillAssignedToAssetsNotification;
use App\User;
use Exception;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;

class TerminatedUsersStillAssignedToAssets extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'TerminatedUsersStillAssignedToAssets:notify';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sending notification with the list of terminated users still assigned to assets.';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        try {
            $webhook = SlackNotificationCredential::where('slug', 'it_channel')->first();

            if (!$webhook) {
                return $this->info('Invalid web hook url');
                Log::error('TerminatedUsersStillAssignedToAssets : Invalid web hook url');
            }

            $assignedStatusId = optional(AssetStatus::where('slug', 'assigned')->first())->id;
            $terminatedUsersWithAssignedAssets = User::where('status', 0)
                ->whereHas('assets', function ($query) use($assignedStatusId) {
                        $query->where('asset_status_id', $assignedStatusId);
                })
                ->pluck('email')
                ->toArray();

            if (!empty($terminatedUsersWithAssignedAssets)) {
                $message = implode(", ", $terminatedUsersWithAssignedAssets);
                Notification::route('slack', $webhook->webhook_url)
                    ->notify(new TerminatedUsersStillAssignedToAssetsNotification($message));
                
                return $this->info(json_encode([
                    'status'  => 'success',
                    'message' => 'Notifications Send successfully'
                ]));
            }

            return $this->info(json_encode([
                'status'  => 'success',
                'message' => 'Nothing to notify'
            ]));

        } catch (Exception $e) {
            return $this->info(json_encode([
                'status'  => 'error',
                'message' => 'Something went wrong. Try again later'
            ]));
            Log::error('TerminatedUsersStillAssignedToAssets : ' . $e->getMessage());
        }
    }
}
